% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/satdad_Rfunctions.R
\name{ticEmp}
\alias{ticEmp}
\title{Empirical tail importance coefficients.}
\usage{
ticEmp(sample, ind = 2, k, sobol = FALSE)
}
\arguments{
\item{sample}{A \code{(n times d)} matrix.}

\item{ind}{A character string among "with.singletons" and "all" (without singletons), or an integer in \eqn{\{2,...,d\}} or a list of subsets from  \eqn{\{1,...,d\}}. The default is \code{ind = 2}, all pairwise coefficients are computed.}

\item{k}{An integer smaller or equal to \code{n}.}

\item{sobol}{A boolean. `FALSE` (the default). If `TRUE`:  the index is normalized by the empirical global variance.}
}
\value{
The function returns a list of two elements:
\itemize{
\item{\code{subsets}} A list of subsets from  \eqn{\{1,...,d\}}.

When \code{ind} is given as an integer, \code{subsets} is the list of subsets from  \eqn{\{1,...,d\}} with cardinality \code{ind}. When \code{ind} is the list, it corresponds to \code{subsets}.

When \code{ind = "with.singletons"}  subsets is the list of all non empty subsets in \eqn{\{1,...,d\}}.

When \code{ind = "all"}   subsets is the list of all subsets in \eqn{\{1,...,d\}} with cardinality larger or equal to 2.

\item{\code{tic}} A vector of tail importance coefficients, or their sobol versions when \code{sobol = "TRUE"}.
}
}
\description{
Computes on a sample the tail  importance coefficients (tic) associated with threshold \code{k}. The value may be renormalized by the empirical global variance (Sobol version).
}
\details{
The theoretical functional decomposition of the variance of the stdf \eqn{\ell} consists in writing \eqn{D(\ell) = \sum_{I \subseteq \{1,...,d\}} D_I(\ell) } where \eqn{D_I(\ell)} measures the variance of \eqn{\ell_I(U_I)} the term associated with subset \eqn{I} in the Hoeffding-Sobol decomposition of \eqn{\ell}
; note that \eqn{U_I} represents  a random vector with independent standard uniform entries. The theoretical tail variance contribution is thus \eqn{D_I(\ell)} and the theoretical tail sobol index is \eqn{S_I(\ell)=\dfrac{D_I(\ell)}{D(\ell)}}.

Here, the function \code{ticEmp} evaluates  \eqn{\hat{D}_{I,k,n}} the empirical counterpart of \eqn{D_I(\ell)} under the option \code{sobol = FALSE}, and \eqn{\hat{S}_{I,k,n}} the empirical counterpart of \eqn{S_I(\ell)} under the option \code{sobol = TRUE}.

Proposition 1 and Theorem 2 of Mercadier and Roustant (2019) furnish their rank-based expressions. For the subset of components \eqn{I},

\eqn{\hat{D}_{I,k,n}=\frac{1}{k^2}\sum_{s=1}^n\sum_{s^\prime=1}^n \prod_{t\in I}(\min(\overline{R}^{(t)}_s,\overline{R}^{(t)}_{s^\prime})-\overline{R}^{(t)}_{s}\overline{R}^{(t)}_{s^\prime}) \prod_{t\notin I} \overline{R}^{(t)}_s\overline{R}^{(t)}_{s^\prime}}

\eqn{\hat{D}_{k,n}=\frac{1}{k^2}\sum_{s=1}^n\sum_{s^\prime=1}^n \prod_{t\in I}\min(\overline{R}^{(t)}_s,\overline{R}^{(t)}_{s^\prime})- \prod_{t\in I}\overline{R}^{(t)}_{s}\overline{R}^{(t)}_{s^\prime}}

and \eqn{\hat{S}_{I,k,n}=\dfrac{\hat{D}_{I,k,n}}{\hat{D}_{k,n}}}

where

\itemize{
\item{} \eqn{k} is the threshold parameter,
\item{} \eqn{n} is the sample size,
\item{} \eqn{X_1,...,X_n} describes the \code{sample}, each \eqn{X_s} is a d-dimensional vector \eqn{X_s^{(t)}} for \eqn{t=1,...,d},
\item{} \eqn{R^{(t)}_s} denotes the rank of \eqn{X^{(t)}_s} among \eqn{X^{(t)}_1, ..., X^{(t)}_n},
\item{} and  \eqn{\overline{R}^{(t)}_s = \min((n- R^{(t)}_s+1)/k,1)}.
}
}
\examples{

## Fix a 5-dimensional asymmetric tail dependence structure
(ds5 <- gen.ds(d = 5))

## Generate a 1000-sample of Mevlog random vectors associated with ds5
sample5 <- rMevlog(n = 1000, ds = ds5)

## Compute empirical tic values according cardinality
res2 <- ticEmp(sample5, ind = 2, k = 100, sobol = TRUE)
res3 <- ticEmp(sample5, ind = 3, k = 100, sobol = TRUE)
res4 <- ticEmp(sample5, ind = 4, k = 100, sobol = TRUE)

## Represent the empirical indices associated with pairs
barplot(res2$tic ~ as.character(res2$subsets), las = 2,
     xlab = "", ylab = "", main = "Tail Sobol Indices (cardinality 2)")

## Represent the empirical indices associated with triplets
barplot(res3$tic ~ as.character(res3$subsets), las = 2,
     xlab = "", ylab = "", main = "Tail Sobol Indices (cardinality 3)")

## Represent the empirical indices associated with quadriplets
barplot(res4$tic ~ as.character(res4$subsets), las = 2,
     xlab = "", ylab ="", main = "Tail Sobol Indices (cardinality 4)")

## Check the sum-to-one constraint of empirical tail Sobol indices
sum(ticEmp(sample5, ind = "with.singletons", k = 100,  sobol = TRUE)$tic)

}
\references{
Mercadier, C. and Roustant, O. (2019)
The tail dependograph.
\emph{Extremes}, \bold{22}, 343--372.
}
\seealso{
\code{\link[satdad]{tic}} and \code{\link[satdad]{tsicEmp}}
}
\author{
Cécile Mercadier (\code{mercadier@math.univ-lyon1.fr})
}
