% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rsvd.R
\name{rsvd}
\alias{rsvd}
\title{Randomized Singular Value Decomposition (rsvd).}
\usage{
rsvd(A, k = NULL, nu = NULL, nv = NULL, p = 10, q = 2,
  sdist = "normal")
}
\arguments{
\item{A}{array_like; \cr
a real/complex \eqn{(m, n)} input matrix (or data frame) to be decomposed.}

\item{k}{integer; \cr
specifies the target rank of the low-rank decomposition. \eqn{k} should satisfy \eqn{k << min(m,n)}.}

\item{nu}{integer, optional; \cr
number of left singular vectors to be returned. \eqn{nu} must be between \eqn{0} and \eqn{k}.}

\item{nv}{integer, optional; \cr
number of right singular vectors to be returned. \eqn{nv} must be between \eqn{0} and \eqn{k}.}

\item{p}{integer, optional; \cr
oversampling parameter (by default \eqn{p=10}).}

\item{q}{integer, optional; \cr
number of additional power iterations (by default \eqn{q=2}).}

\item{sdist}{string \eqn{c( 'unif', 'normal', 'rademacher')}, optional; \cr
specifies the sampling distribution of the random test matrix: \cr
        \eqn{'unif'} :  Uniform `[-1,1]`. \cr
        \eqn{'normal}' (default) : Normal `~N(0,1)`. \cr
        \eqn{'rademacher'} : Rademacher random variates. \cr}
}
\value{
\code{rsvd} returns a list containing the following three components:
\item{d}{  array_like; \cr
          singular values; vector of length \eqn{(k)}.
}

\item{u}{  array_like; \cr
          left singular vectors; \eqn{(m, k)} or \eqn{(m, nu)} dimensional array.
}

\item{v}{  array_like; \cr
          right singular vectors; \eqn{(n, k)} or \eqn{(n, nv)} dimensional array. \cr
}
}
\description{
The randomized SVD computes the near-optimal low-rank approximation of a rectangular matrix
using a fast probablistic algorithm.
}
\note{
The singular vectors are not unique and only defined up to sign
(a constant of modulus one in the complex case). If a left singular vector
has its sign changed, changing the sign of the corresponding right vector
gives an equivalent decomposition.
}
\examples{
library('rsvd')

# Create a n x n Hilbert matrix of order n,
# with entries H[i,j] = 1 / (i + j + 1).
hilbert <- function(n) { i <- 1:n; 1 / outer(i - 1, i, "+") }
H <- hilbert(n=50)

# Low-rank (k=10) matrix approximation using rsvd
k=10
s <- rsvd(H, k=k)
Hre <- s$u \%*\% diag(s$d) \%*\% t(s$v) # matrix approximation
print(100 * norm( H - Hre, 'F') / norm( H,'F')) # percentage error
# Compare to truncated base svd
s <- svd(H)
Hre <- s$u[,1:k] \%*\% diag(s$d[1:k]) \%*\% t(s$v[,1:k]) # matrix approximation
print(100 * norm( H - Hre, 'F') / norm( H,'F')) # percentage error

}
\references{
\itemize{
  \item  [1] N. Halko, P. Martinsson, and J. Tropp.
         "Finding structure with randomness: probabilistic
         algorithms for constructing approximate matrix
         decompositions" (2009).
         (available at arXiv \url{http://arxiv.org/abs/0909.4061}).
  \item  [2] N. B. Erichson, S. Voronin, S. Brunton, J. N. Kutz.
         "Randomized matrix decompositions using R" (2016).
         (available at `arXiv \url{http://arxiv.org/abs/1608.02148}).
}
}
\seealso{
\code{\link{svd}}, \code{\link{rpca}}
}
\author{
N. Benjamin Erichson, \email{erichson@uw.edu}
}
