% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anova_test.R
\name{anova_test}
\alias{anova_test}
\alias{get_anova_table}
\alias{print.anova_test}
\alias{plot.anova_test}
\title{Anova Test}
\usage{
anova_test(data, formula, dv, wid, between, within, covariate,
  type = NULL, effect.size = "ges", error = NULL,
  white.adjust = FALSE, observed = NULL, detailed = FALSE)

get_anova_table(x, correction = c("auto", "GG", "HF", "none"))

\method{print}{anova_test}(x, ...)

\method{plot}{anova_test}(x, ...)
}
\arguments{
\item{data}{a data.frame or a model to be analyzed.}

\item{formula}{a formula specifying the ANOVA model similar to
 \link[stats]{aov}. Can be of the form \code{y ~ group} where \code{y} is a
 numeric variable giving the data values and \code{group} is a factor with
 one or multiple levels giving the corresponding groups. For example,
 \code{formula = TP53 ~ cancer_group}.

 Examples of supported formula include: \itemize{ \item Between-Ss ANOVA
 (independent measures ANOVA): \code{y ~ b1*b2} \item Within-Ss ANOVA
 (repeated measures ANOVA): \code{y ~ w1*w2 + Error(id/(w1*w2))} \item Mixed
 ANOVA: \code{y ~ b1*b2*w1 + Error(id/w1)} }

 If the formula doesn't contain any within vars, a linear model is directly
 fitted and  passed to the ANOVA function. For repeated designs, the ANOVA
 variables are parsed from the formula.}

\item{dv}{(numeric) dependent variable name.}

\item{wid}{(factor) column name containing individuals/subjects identifier.
Should be unique per individual.}

\item{between}{(optional) between-subject factor variables.}

\item{within}{(optional) within-subjects factor variables}

\item{covariate}{(optional) covariate names (for ANCOVA)}

\item{type}{the type of sums of squares for ANOVA. Allowed values are either
1, 2 or 3. \code{type = 2} is the default because this will yield identical
ANOVA results as type = 1 when data are balanced but type = 2 will
additionally yield various assumption tests where appropriate. When the data
are unbalanced the \code{type = 3} is used by popular commercial softwares
including SPSS.}

\item{effect.size}{the effect size to compute and to show in the ANOVA
results. Allowed values can be either "ges" (generalized eta squared) or
"pes" (partial eta squared) or both. Default is "ges".}

\item{error}{(optional) for a linear model, an lm model object from which the
overall error sum of squares and degrees of freedom are to be calculated.
Read more in \code{\link[car]{Anova}()} documentation.}

\item{white.adjust}{Default is FALSE. If TRUE, heteroscedasticity correction
is applied to the coefficient of covariance matrix. Used only for
independent measures ANOVA.}

\item{observed}{Variables that are observed (i.e, measured) as compared to
experimentally manipulated. The default effect size reported (generalized
eta-squared) requires correct specification of the observed variables.}

\item{detailed}{If TRUE, returns extra information (sums of squares columns,
intercept row, etc.) in the ANOVA table.}

\item{x}{an object of class \code{anova_test}}

\item{correction}{character. Used only in repeated measures ANOVA test to
specify which correction of the degrees of freedom should be reported for
the within-subject factors. Possible values are:
\itemize{
\item{"GG"}: applies Greenhouse-Geisser correction to all within-subjects factors even if the assumption of sphericity is met (i.e., Mauchly's test is not significant, p > 0.05).
\item{"HF"}: applies Hyunh-Feldt correction to all within-subjects factors even if the assumption of sphericity is met,
\item{"none"}: returns the ANOVA table without any correction and
\item{"auto"}: apply automatically GG correction to only within-subjects factors violating the sphericity assumption (i.e., Mauchly's test p-value is significant, p <= 0.05).
}}

\item{...}{additional arguments}
}
\value{
return an object of class \code{anova_test} a data frame containing
 the ANOVA table for independent measures ANOVA.

 However, for repeated/mixed measures ANOVA, a list containing the following
 components are returned: ANOVA table, Mauchly's Test for Sphericity,
 Sphericity Corrections. These table are described more in the documentation
 of the function \code{\link{anova_summary}()}.

 The \strong{returned object has an attribute} called \code{args}, which is a
 list holding the arguments used to fit the ANOVA model, including: data, dv,
 within, between, type, model, etc.
}
\description{
Provides a pipe-friendly framework to perform different types of
 ANOVA tests, including: \itemize{ \item \strong{\href{https://www.datanovia.com/en/lessons/anova-in-r/}{Independent measures ANOVA}}:
 between-Subjects designs, \item \strong{\href{https://www.datanovia.com/en/lessons/repeated-measures-anova-in-r/}{Repeated measures ANOVA}}:
 within-Subjects designs \item \strong{\href{https://www.datanovia.com/en/lessons/mixed-anova-in-r/}{Mixed ANOVA}}: Mixed within within- and
 between-Subjects designs, also known as split-plot ANOVA and \item \strong{\href{https://www.datanovia.com/en/lessons/ancova-in-r/}{ANCOVA: Analysis of Covariance}}. }

 The function is an easy to use wrapper around \code{\link[car]{Anova}()} and
 \code{\link[stats]{aov}()}. It makes ANOVA computation handy in R and It's
 highly flexible: can support model and formula as input. Variables can be
 also specified as character vector using the arguments \code{dv, wid,
 between, within, covariate}.

 The results include ANOVA table, generalized effect size and some assumption
 checks.
}
\section{Functions}{
\itemize{
\item \code{anova_test}: perform anova test

\item \code{get_anova_table}: extract anova table from an object of class
\code{anova_test}. When within-subject factors are present, either
sphericity corrected or uncorrected degrees of freedom can be reported.
}}

\examples{
# Load data
#:::::::::::::::::::::::::::::::::::::::
data("ToothGrowth")
df <- ToothGrowth

# One-way ANOVA test
#:::::::::::::::::::::::::::::::::::::::::
df \%>\% anova_test(len ~ dose)

# Grouped One-way ANOVA test
#:::::::::::::::::::::::::::::::::::::::::
df \%>\%
  group_by(supp) \%>\%
  anova_test(len ~ dose)

# Two-way ANOVA test
#:::::::::::::::::::::::::::::::::::::::::
df \%>\% anova_test(len ~ supp*dose)

# Two-way repeated measures ANOVA
#:::::::::::::::::::::::::::::::::::::::::
df$id <- rep(1:10, 6) # Add individuals id
# Use formula
\donttest{
df \%>\% anova_test(len ~ supp*dose + Error(id/(supp*dose)))
}


# or use character vector
df \%>\% anova_test(dv = len, wid = id, within = c(supp, dose))

# Extract ANOVA table and apply correction
#:::::::::::::::::::::::::::::::::::::::::
res.aov <- df \%>\% anova_test(dv = len, wid = id, within = c(supp, dose))
get_anova_table(res.aov, correction = "GG")


# Use model as arguments
#:::::::::::::::::::::::::::::::::::::::::
.my.model <- lm(yield ~ block + N*P*K, npk)
anova_test(.my.model)


}
\seealso{
\code{\link{anova_summary}()}, \code{\link{factorial_design}()}
}
\author{
Alboukadel Kassambara, \email{alboukadel.kassambara@gmail.com}
}
