# --------------------------------------
# Author: Andreas Alfons
#         Erasmus Universiteit Rotterdam
# --------------------------------------

#' Plot (robust) mediation analysis results
#'
#' Visualize results from (robust) mediation analysis.
#'
#' The \code{"\link{fit_mediation}"} method is a wrapper for
#' \code{\link{ellipse_plot}()}.
#'
#' The \code{"\link{test_mediation}"} method calls \code{\link{ci_plot}()},
#' \code{\link{density_plot}()}, or \code{\link{ellipse_plot}()}, depending
#' on the argument \code{which}.
#'
#' @name plot-methods
#'
#' @param object,x  an object inheriting from class
#' \code{"\link{fit_mediation}"} or \code{"\link{test_mediation}"} containing
#' results from (robust) mediation analysis.
#' @param which  a character string specifying which plot to produce.
#' Possible values are \code{"ci"} for a dot plot of selected effects
#' together with confidence intervals (see \code{\link{ci_plot}()}),
#' \code{"density"} for a density plot of the indirect effect(s) (see
#' \code{\link{density_plot}()}), or \code{"ellipse"} for a diagnostic plot
#' of the data together with a tolerance ellipse (see
#' \code{\link{ellipse_plot}()}).
#' @param \dots  additional arguments to be passed down.
#'
#' @return An object of class \code{"\link[ggplot2]{ggplot}"}.
#'
#' @author Andreas Alfons
#'
#' @seealso
#' \code{\link{fit_mediation}()}, \code{\link{test_mediation}()}
#'
#' \code{\link{ci_plot}()}, \code{\link{density_plot}()},
#' \code{\link{ellipse_plot}()}
#'
#' @examples
#' data("BSG2014")
#'
#' # run fast and robust bootstrap test
#' robust_boot <- test_mediation(BSG2014,
#'                               x = "ValueDiversity",
#'                               y = "TeamCommitment",
#'                               m = "TaskConflict",
#'                               robust = TRUE)
#'
#' # create plots for robust bootstrap test
#' plot(robust_boot, which = "ci")
#' plot(robust_boot, which = "density")
#' plot(robust_boot, which = "ellipse")
#'
#' @keywords hplot
#'
#' @import ggplot2

NULL


## internal function for plotting
plot_internal <- function(object, which = c("ci", "density", "ellipse"), ...) {
  # initializations
  which <- match.arg(which)
  # call selected plot function
  if (which == "ci") ci_plot(object, ...)
  else if (which == "density") density_plot(object, ...)
  else if (which == "ellipse") ellipse_plot(object, ...)
  else stop("type of plot not implemented")  # shouldn't happen
}


#' @rdname plot-methods
#' @method autoplot fit_mediation
#' @export

autoplot.fit_mediation <- function(object, ...) ellipse_plot(object, ...)


#' @rdname plot-methods
#' @method autoplot test_mediation
#' @export

autoplot.test_mediation <- function(object,
                                    which = c("ci", "density", "ellipse"),
                                    ...) {
  plot_internal(object, which = which, ...)
}


#' @rdname plot-methods
#' @method plot fit_mediation
#' @export

plot.fit_mediation <- function(x, ...) ellipse_plot(x, ...)


#' @rdname plot-methods
#' @method plot test_mediation
#' @export

plot.test_mediation <- function(x, which = c("ci", "density", "ellipse"),
                                ...) {
  plot_internal(x, which = which, ...)
}
