% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/priors.R
\name{set_prior}
\alias{set_prior}
\title{Construction of prior distributions for extreme value model parameters}
\usage{
set_prior(
  prior = c("norm", "loglognorm", "mdi", "flat", "flatflat", "jeffreys", "beta", "prob",
    "quant"),
  model = c("gev", "gp", "pp", "os"),
  ...
)
}
\arguments{
\item{prior}{Either
\itemize{
  \item An R function, or a pointer to a user-supplied compiled
  C++ function, that returns the value of the log of the prior density
  (see \strong{Examples}), or
  \item A character string giving the name of the prior.
    See \strong{Details} for a list of priors available for each model.
}}

\item{model}{A character string.  If \code{prior} is a character string
then \code{model} gives the extreme value model to be used.  Using
either \code{model = "gev"}, \code{model = "pp"} or
\code{model = "os"} will result in the same (GEV) parameterisation.
If \code{prior} is a function then the value of \code{model} is stored
so that in the subsequent call to \code{rpost}, consistency of the
prior and extreme value model parameterisations can be checked.}

\item{...}{Further arguments to be passed to the user-supplied or
in-built prior function.  For details of the latter see \strong{Details}
and/or the relevant underlying function: \code{\link{gp_norm}},
\code{\link{gp_mdi}}, \code{\link{gp_flat}}, \code{\link{gp_flatflat}},
\code{\link{gp_jeffreys}}, \code{\link{gp_beta}},
\code{\link{gev_norm}}, \code{\link{gev_loglognorm}},
\code{\link{gev_mdi}}, \code{\link{gev_flat}}, \code{\link{gev_flatflat}},
\code{\link{gev_beta}}, \code{\link{gev_prob}}, \code{\link{gev_quant}}.
All these priors have the arguments \code{min_xi} (prior lower bound on
\eqn{\xi}) and \code{max_xi} (prior upper bound on \eqn{\xi}).}
}
\value{
A list with class \code{"evprior"}.  The first component is the
  input prior, i.e. either the name of the prior or a user-supplied
  function.  The remaining components contain the numeric values of any
  hyperparameters in the prior.
}
\description{
Constructs a prior distribution for use as the argument \code{prior} in
\code{\link{rpost}} and \code{\link{rpost_rcpp}}.  The user can either
specify their own prior function, returning the \strong{log} of the prior
density, (using an R function or an external pointer to a compiled C++
function) and arguments for hyperparameters or choose from a list of
in-built model-specific priors.  Note that the arguments
\code{model = "gev"}, \code{model = "pp"} and \code{model =="os"} are
equivalent because a prior is specified is the GEV parameterisation in each
of these cases.
Note also that for \code{model = "pp"} the prior GEV parameterisation
relates to the value of \code{noy} subsequently supplied to
\code{\link{rpost}} or \code{\link{rpost_rcpp}}.
The argument \code{model} is used for consistency with \code{rpost}.
}
\details{
Of the in-built named priors available in revdbayes only
  those specified using \code{prior = "prob"} (\code{\link{gev_prob}}),
  \code{prior = "quant"} (\code{\link{gev_quant}})
  \code{prior = "norm"} (\code{\link{gev_norm}}) or
  \code{prior = "loglognorm"} (\code{\link{gev_loglognorm}}) are proper.
  If \code{model = "gev"} these priors are equivalent to priors available
  in the evdbayes package, namely \code{prior.prob},
  \code{prior.quant}, \code{prior.norm} and \code{prior.loglognorm}.

  The other in-built priors are improper, that is, the integral of the
  prior function over its support is not finite.  Such priors do not
  necessarily result in a proper posterior distribution. Northrop and
  Attalides (2016) consider the issue of posterior propriety in Bayesian
  extreme value analyses.  In most of improper priors below the prior for
  the scale parameter \eqn{\sigma} is taken to be \eqn{1/\sigma},
  i.e. a flat prior for \eqn{\log \sigma}{log \sigma}.  Here we denote the
  scale parameter of the GP distribution by \eqn{\sigma}, whereas we use
  \eqn{\sigma_u} in the revdbayes vignette.

  For all in-built priors the arguments \code{min_xi} and \code{max_xi} may
  be supplied by the user.  The prior density is set to zero for any value
  of the shape parameter \eqn{\xi} that is outside
  (\code{min_xi}, \code{max_xi}).  This will override the default values
  of \code{min_xi} and \code{max_xi} in the named priors detailed above.

  \strong{Extreme value priors.} It is typical to use either
  \code{prior = "prob"} (\code{\link{gev_prob}}) or
  \code{prior = "quant"} (\code{\link{gev_quant}}) to set an informative
  prior and one of the other prior (or a user-supplied function) otherwise.
  The names of the in-built extreme value priors set using \code{prior}
  and details of hyperparameters are:
\itemize{
  \item \code{"prob"}.  A prior for GEV parameters \eqn{(\mu, \sigma, \xi)}
    based on Crowder (1992).  See \code{\link{gev_prob}} for details.
    See also Northrop et al. (2017) and Stephenson (2016).
  \item \code{"quant"}.  A prior for GEV parameters \eqn{(\mu, \sigma, \xi)}
    based on Coles and Tawn (1996). See \code{\link{gev_quant}} for details.
  \item \code{"norm"}.

  For \code{model = "gp"}:
    (\eqn{\log \sigma, \xi}{log \sigma, \xi}), is bivariate normal
    with mean \code{mean} (a numeric vector of length 2) and covariance
    matrix \code{cov} (a symmetric positive definite 2 by 2 matrix).

  For \code{model = "gev"}:
    (\eqn{\mu, \log \sigma, \xi}{\mu, log \sigma, \xi}), is trivariate
    normal with mean \code{mean} (a numeric vector of length 3) and
    covariance matrix \code{cov} (a symmetric positive definite 3 by 3
    matrix).

  \item \code{"loglognorm"}.  For \code{model = "gev"} only:
    (\eqn{\log \mu, \log \sigma, \xi}{log \mu, log \sigma, \xi}), is
    trivariate normal with mean \code{mean} (a numeric vector of length 3)
    and covariance matrix \code{cov} (a symmetric positive definite 3 by 3
    matrix).

  \item \code{"mdi"}.

  For \code{model = "gp"}: (an extended version
    of) the maximal data information (MDI) prior, that is,
    \deqn{\pi(\sigma, \xi) = \sigma^{-1} \exp[-a(\xi + 1)], {\rm ~for~}
    \sigma > 0, \xi \geq -1, a \geq 0.}{%
    \pi(\sigma, \xi) = (1/ \sigma) exp[- a (\xi + 1)], for
    \sigma > 0, \xi >= -1, a >= 0.}
    The value of \eqn{a} is set using the argument \code{a}.  The default
    value is \eqn{a = 1}, which gives the MDI prior.

    For \code{model = "gev"}: (an extended version
    of) the maximal data information (MDI) prior, that is,
    \deqn{\pi(\mu, \sigma, \xi) = \sigma^{-1} \exp[-a(\xi + 1)],
    {\rm ~for~} \sigma > 0, \xi \geq -1, a \geq 0.}{%
    \pi(\mu, \sigma, \xi) = (1/ \sigma) exp[- a (\xi + 1)], for
    \sigma > 0, \xi >= -1, a >= 0.}
    The value of \eqn{a} is set using the argument \code{a}.  The default
    value is \eqn{a = \gamma}, where \eqn{\gamma = 0.57721} is Euler's
    constant, which gives the MDI prior.

    For each of these cases \eqn{\xi} must be is bounded below
    \emph{a priori} for the posterior to be proper
    (Northrop and Attalides, 2016).  An argument for the
    bound \eqn{\xi \geq -1}{\xi >= -1} is that for \eqn{\xi < -1} the
    GP (GEV) likelihood is unbounded above as \eqn{-\sigma/\xi}
    (\eqn{\mu - \sigma/\xi})) approaches the sample maximum.  In
    maximum likelihood estimation of GP parameters (Grimshaw, 1993)
    and GEV parameters a local maximum of the likelihood
    is sought on the region
    \eqn{\sigma > 0, \xi \geq -1}{\sigma > 0, \xi >= -1}.

  \item\code{"flat"}.

    For \code{model = "gp"}: a flat prior for
    \eqn{\xi} (and for \eqn{\log \sigma}{log \sigma}):
    \deqn{\pi(\sigma, \xi) = \sigma^{-1}, {\rm ~for~} \sigma > 0.}{%
          \pi(\sigma, \xi) = (1/ \sigma), for \sigma > 0.}

    For \code{model = "gev"}: a flat prior for
    \eqn{\xi} (and for \eqn{\mu} and \eqn{\log \sigma}{log \sigma}):
    \deqn{\pi(\mu, \sigma, \xi) = \sigma^{-1}, {\rm ~for~} \sigma > 0.}{%
          \pi(\mu, \sigma, \xi) = (1/ \sigma), for \sigma > 0.}

  \item \code{"flatflat"}.

    For \code{model = "gp"}: flat priors for
    \eqn{\sigma} and \eqn{\xi}:
    \deqn{\pi(\sigma, \xi) = 1, {\rm ~for~} \sigma > 0.}{%
          \pi(\sigma, \xi) = 1, for \sigma > 0.}

    For \code{model = "gev"}: flat priors for \eqn{\mu}, \eqn{\sigma}
    and \eqn{\xi}:
    \deqn{\pi(\mu, \sigma, \xi) = 1, {\rm ~for~} \sigma > 0.}{%
          \pi(\mu, \sigma, \xi) = 1, for \sigma > 0.}

    Therefore, the posterior is proportional to the likelihood.

  \item \code{"jeffreys"}.  For \code{model = "gp"} only: the Jeffreys
    prior (Castellanos and Cabras, 2007):
    \deqn{\pi(\sigma, \xi) = \sigma^{-1}(1+\xi)^{-1}(1+2\xi)^{-1/2},
      {\rm ~for~} \sigma > 0, \xi > -1 / 2.}{%
      \pi(\sigma, \xi) = 1/ [\sigma (1+\xi) \sqrt(1+2\xi)],
      for \sigma > 0, \xi > -1 / 2.}

    In the GEV case the Jeffreys prior doesn't yield a proper posterior
    for any sample size.  See Northrop and Attalides (2016) for details.

  \item \code{"beta"}.
    For \code{model = "gp"}: a beta-type(p, q)
    prior is used for xi on the interval (\code{min_xi}, \code{max_xi}):
    \deqn{\pi(\sigma, \xi) = \sigma^{-1} (\xi - {\min}_{\xi}) ^ {p-1}
          ({\max}_{\xi} - \xi) ^ {q-1}, {\rm ~for~}
          {\min}_{\xi} < \xi < {\max}_{\xi}.}{%
          \pi(\sigma, \xi) = (1/\sigma) (\xi - min_xi) ^ (p-1)
          (max_xi - \xi) ^ (q-1), for min_xi < \xi < max_xi.}

    For \code{model = "gev"}: similarly ...
    \deqn{\pi(\mu, \sigma, \xi) = \sigma^{-1} (\xi - {\min}_{\xi}) ^ {p-1}
          ({\max}_{\xi} - \xi) ^ {q-1}, {\rm ~for~}
          {\min}_{\xi} < \xi < {\max}_{\xi}.}{%
          \pi(\mu, \sigma, \xi) = (1/\sigma) (\xi - min_xi) ^ (p-1)
          (max_xi - \xi) ^ (q-1), for min_xi < \xi < max_xi.}

    The argument \code{pq} is a vector containing \code{c(p,q)}.
    The default settings for this prior are \code{p = 6, q = 9} and
    \code{min_xi = -1/2, max_xi = 1/2}, which corresponds to the
    prior for \eqn{\xi} proposed in Martins and Stedinger (2000, 2001).

}
}
\examples{
# Normal prior for GEV parameters (mu, log(sigma), xi).
mat <- diag(c(10000, 10000, 100))
pn <- set_prior(prior = "norm", model = "gev", mean = c(0,0,0), cov = mat)
pn

# Prior for GP parameters with flat prior for xi on (-1, infinity).
fp <- set_prior(prior = "flat", model = "gp", min_xi = -1)
fp

# A user-defined prior (see the vignette for details).
u_prior_fn <- function(x, ab) {
  if (x[1] <= 0 | x[2] <= -1 | x[2] >= 1) {
    return(-Inf)
  }
  return(-log(x[1]) + (ab[1] - 1) * log(1 + x[2]) +
         (ab[2] - 1) * log(1 - x[2]))
}
up <- set_prior(prior = u_prior_fn, ab = c(2, 2), model = "gp")

# A user-defined prior using a pointer to a C++ function
ptr_gp_flat <- create_prior_xptr("gp_flat")
u_prior_ptr <- set_prior(prior = ptr_gp_flat, model = "gp")
}
\references{
Castellanos, E. M. and Cabras, S. (2007) A default Bayesian
  procedure for the generalized Pareto distribution.
  \emph{Journal of Statistical Planning and Inference} \strong{137(2)},
  473-483. \doi{10.1016/j.jspi.2006.01.006}.

Coles, S. G. and Tawn, J. A. (1996) A Bayesian analysis of
  extreme rainfall data. \emph{Appl. Statist.}, \strong{45}, 463-478.

Crowder, M. (1992) Bayesian priors based on parameter
  transformation using the distribution function
  \emph{Ann. Inst. Statist. Math.}, \strong{44}, 405-416.
  \url{https://link.springer.com/article/10.1007/BF00050695}.

Grimshaw, S. D. (1993) Computing Maximum Likelihood Estimates
  for the Generalized Pareto Distribution.  \emph{Technometrics},
  \strong{35(2)}, 185-191.
  \doi{10.1080/00401706.1993.10485040}.

Hosking, J. R. M. and Wallis, J. R. (1987) Parameter and
  Quantile Estimation for the Generalized Pareto Distribution.
  \emph{Technometrics}, \strong{29(3)}, 339-349.
  \doi{10.2307/1269343}.

Martins, E. S. and Stedinger, J. R. (2000) Generalized maximum
  likelihood generalized extreme value quantile estimators for hydrologic
  data, \emph{Water Resources Research}, \strong{36(3)}, 737-744.
  \doi{10.1029/1999WR900330}.

Martins, E. S. and Stedinger, J. R. (2001) Generalized maximum
  likelihood Pareto-Poisson estimators for partial duration series,
  \emph{Water Resources Research}, \strong{37(10)}, 2551-2557.
  \doi{10.1029/2001WR000367}.

Northrop, P.J. and Attalides, N. (2016) Posterior propriety in
  Bayesian extreme value analyses using reference priors
  \emph{Statistica Sinica}, \strong{26}(2), 721--743
  \doi{10.5705/ss.2014.034}.

Northrop, P. J., Attalides, N. and Jonathan, P. (2017)
  Cross-validatory extreme value threshold selection and uncertainty
  with application to ocean storm severity.
  \emph{Journal of the Royal Statistical Society Series C: Applied
  Statistics}, \strong{66}(1), 93-120.
  \doi{10.1111/rssc.12159}

Stephenson, A. (2016) Bayesian inference for extreme value
  modelling.  In \emph{Extreme Value Modeling and Risk Analysis: Methods
  and Applications} (eds D. K. Dey and J. Yan), 257-280, Chapman and Hall,
  London. \doi{10.1201/b19721}.
}
\seealso{
\code{\link{rpost}} and \code{\link{rpost_rcpp}} for sampling
  from an extreme value posterior distribution.

\code{\link{create_prior_xptr}} for creating an external
  pointer to a C++ function to evaluate the log-prior density.

\code{\link{rprior_prob}} and \code{\link{rprior_quant}} for
  sampling from informative prior distributions for GEV parameters.

\code{\link{gp_norm}}, \code{\link{gp_mdi}},
  \code{\link{gp_flat}}, \code{\link{gp_flatflat}},
  \code{\link{gp_jeffreys}}, \code{\link{gp_beta}} to see the arguments
  for priors for GP parameters.

\code{\link{gev_norm}}, \code{\link{gev_loglognorm}},
  \code{\link{gev_mdi}}, \code{\link{gev_flat}}, \code{\link{gev_flatflat}},
  \code{\link{gev_beta}}, \code{\link{gev_prob}}, \code{\link{gev_quant}}
  to see the arguments for priors for GEV parameters.
}
