% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oxy_crit.R
\name{oxy_crit}
\alias{oxy_crit}
\title{Calculate critical oxygen values, such as PCrit}
\usage{
oxy_crit(
  x,
  method = "bsr",
  time = NULL,
  oxygen = NULL,
  rate = NULL,
  width = 0.1,
  parallel = FALSE,
  thin = 5000,
  plot = TRUE,
  ...
)
}
\arguments{
\item{x}{object of class \code{inspect} or a \code{data.frame} containing either paired
oxygen~time values, or paired rate~oxygen values. See Details.}

\item{method}{string. Defaults to \code{"bsr"}. Critical oxygen value analysis
method. Either \code{"bsr"} or \code{"segmented"}. See Details.}

\item{time}{integer. Defaults to 1. Specifies column number of the time data.}

\item{oxygen}{integer. Defaults to 2. Specifies column number of the oxygen
data.}

\item{rate}{integer. Defaults to NULL. Specifies column number of the rate
data.}

\item{width}{numeric value between 0 and 1 representing proportion of the
total data length. Determines the width of the rolling regression used to
determine the rolling rate and the rolling mean of oxygen values the rate
is paired with. Defaults to 0.1, representing 10\% of total rows.}

\item{parallel}{logical. Defaults to FALSE. Enables parallel processing for
computationally intensive analyses of large datasets.}

\item{thin}{integer. Defaults to 5000. Number of rows to subsample \code{x} data
to before running \code{"bsr"} analysis. No effect on datasets smaller than this
value or with \code{"segmented"} method. To perform no subsampling enter as
\code{NULL}. See Details.}

\item{plot}{logical. Defaults to TRUE.}

\item{...}{Allows additional plotting controls to be passed, such as \code{legend = FALSE}, \code{quiet = TRUE}, \code{rate.rev = FALSE}, and \code{panel}. See Plotting
section.}
}
\value{
Output is a \code{list} object of class \code{oxy_crit} containing input
parameters and data, various summary data, metadata, and the primary output
of interest \verb{$crit}, which is the critical oxygen value in the units of the
oxygen data as entered. This can be converted to additional units using
\code{\link[=convert_DO]{convert_DO()}}. Note, if the Broken-Stick analysis (\code{method == "bsr"})
has been used, \verb{$crit} will contain two results; \verb{$crit.intercept} and
\verb{$crit.midpoint}. For full explanation of the difference between these see
Yeager & Ultsch (1989), however they are generally very close in value.
}
\description{
A function to calculate critical oxygen values, the oxygen tension or
concentration below which an uptake rate transitions from independent to
dependent on the oxygen supply, typically known as \emph{PCrit}.
}
\details{
In earlier versions of \code{respR}, this function was known as \code{pcrit} or
\code{calc_pcrit}. It was renamed to avoid conflicts with functions of the same
name in another package, and also because technically the \emph{P} in  \emph{PCrit}
stands for the partial \emph{pressure} of oxygen. Since the function returns the
value in the units of the data as entered, whether they are concentration or
pressure units, this terminology can be technically in error. Instead, for
the purposes of the documentation we refer to this as the \emph{Critical Oxygen
Value}, or "\emph{COV}". If the units of oxygen are partial pressure units (e.g.
kPa), this is equivalent to PCrit, otherwise they should be reported with
this in mind.
\subsection{Methods}{

The \code{oxy_crit()} function provides two methods (one of which outputs two
results) to calculate the \emph{COV}. These are selected using the \code{method} input.
\subsection{Broken Stick Regression:    \code{method = "bsr"}}{

This is the default method, adapted from the “Broken-Stick” regression
(\emph{BSR}) approach, of Yeager & Ultsch (1989), in which two segments of the
data are iteratively fitted and the intersection with the smallest sum of the
residual sum of squares between the two linear models is the estimated \emph{COV}.
Two slightly different ways of reporting this breakpoint are detailed by
Yeager & Ultsch (1989); the \emph{intercept} and \emph{midpoint}. These are usually
very close in value, and the function returns both.

The \code{thin} input influences the \emph{BSR} analysis. The method is very
computationally intensive, so to speed up analyses the \code{thin} input will
subsample datasets longer than this input to this number or rows before
analysis. The default value of 5000 has in testing provided a good balance
between speed and results accuracy and repeatability. However, results may
vary with different datasets, so users should experiment with varying the
value. To perform no subsampling and use the entire dataset enter \code{thin = NULL}. It has no effect on datasets shorter than the \code{thin} input.
}

\subsection{Segmented Regression:    \code{method = "segmented"}}{

The second method is a wrapper for the "Segmented" regression approach,
available as part of the \code{segmented} R package (Muggeo 2008), which estimates
the \emph{COV} by iteratively fitting two intersecting models and selecting the
value that minimises the “gap” between the fitted lines.
}

}

\subsection{Inputs}{

The data input \code{x} should be an \code{inspect} object or \code{data.frame} containing
oxygen~time data, or a \code{data.frame} containing rate~oxygen data.
\subsection{Oxygen ~ Time data}{

This is the typical input, where a timeseries of oxygen concentrations or
partial pressures against time has been recorded, generally down to a very
low value of oxygen. A column of \code{time} and a column of \code{oxygen} should be
specified. The function defaults to \code{time = 1} and \code{oxygen = 2} if no other
inputs are entered. If an \code{inspect} object is entered as the \code{x} input, the
data frame is extracted automatically and column identifiers are not required
since these were already entered in \code{inspect}. Note, if multiple \code{oxygen}
columns were entered in \code{inspect} only the first entered one will be used in
\code{oxy_crit}.

To calculate the \emph{COV}, the function requires data in the form of oxygen
uptake rate against oxygen value. Therefore, the function performs a rolling
regression on the oxygen~time data to determine rates, and pairs these
against a rolling mean of the oxygen data. The function then performs the
selected analysis \code{method} on these data. The width of the rolling regression
and rolling mean is determined by the \code{width} input. The default is 0.1,
representing 10\% of the length of the data. This performs well in testing,
however performance may vary with data that has abrupt changes in rate, or is
particularly noisy. Users should experiment with different \code{width} values to
see how it affects results, and report this with their results and analysis
parameters.
}

\subsection{Rate ~ Oxygen data}{

Alternatively, if existing rolling oxygen uptake rates have been calculated,
and have appropriate paired oxygen concentration or partial pressure values,
these can be entered with the \code{rate} and \code{oxygen} inputs specifying the
respective columns. In this case the function performs the selected analysis
\code{method} on these data directly without any processing. The \code{width} input in
this case is not relevant and is ignored.

This option can only be used with data frame \code{x} inputs. Note, other columns
such as time data may be present in the input, but are not required so need
not be specified.
}

}

\subsection{Plot}{

A plot is produced (provided \code{plot = TRUE}) of the input data and results.
The top panel is the input data, either the oxygen~time timeseries, or the
rate~oxygen series, depending on what was entered in \code{x}. If the former, the
critical oxygen value is indicated by a horizontal line, or two lines in the
case of the Broken-Stick analysis. Note, since the two \emph{BSR} results are
usually close in value these may overlay each other.

The bottom plot is the rate~oxygen series upon which the analysis was
conducted, either as input or as calculated. Critical oxygen values are
indicated by vertical lines, and regression fits upon which the analysis was
based by black dashed lines.

Note, that in \code{respR} oxygen uptake rates are negative since they represent a
negative slope of oxygen against time, therefore by default rates are plotted
on a reverse y-axis so higher rates appear higher on the plot. If analysing
already calculated rates which are positive values this behaviour can be
reversed by passing \code{rate.rev = FALSE} in either the main function call or
when calling \code{plot()} on the output object. There is no issue with using
positive rate values; they will give identical critical value results in the
analysis.
\subsection{Additional plotting options}{

If the legend obscures parts of the plot they can be suppressed using \code{legend = FALSE}. Suppress console output messages with \code{quiet = TRUE}. Each panel
can be plotted on its own using \code{panel = 1} or \code{panel = 2}. If using
already-calculated, positive rate values to identify critical oxygen values,
the y-axis of the rolling rate plot can be plotted \emph{not} reversed by passing
\code{rate.rev = FALSE} These inputs can be passed in either the main \code{oxy_crit}
call or when calling \code{plot()} on the output object. If axis labels
(particularly y-axis) are difficult to read, \code{las = 2} can be passed to make
axis labels horizontal, and \code{oma} (outer margins, default \code{oma = c(0.4, 1, 1.5, 0.4)}), and \code{mai} (inner margins, default \code{mai = c(0.3, 0.15, 0.35, 0.15)}) used to adjust plot margins.
}

}

\subsection{S3 Generic Functions}{

Saved output objects can be used in the generic S3 functions \code{print()} and
\code{summary()}.
\itemize{
\item \code{print()}: prints the critical oxygen value for the particular \code{method}
used.
\item \code{summary()}: prints critical oxygen value, plus additional coefficients and
metadata for the particular \code{method} used. See Yeager & Ultsch (1989) and
Muggeo (2008) for what these represent. The summary can be exported as a
separate data frame by passing \code{export = TRUE}.
}
}

\subsection{More}{

For additional help, documentation, vignettes, and more visit the \code{respR}
website at \url{https://januarharianto.github.io/respR/}
}
}
\examples{
\donttest{
## Run on oxygen~time data.frame with default inputs
oxy_crit(squid.rd)

## Try a lower 'thin' input to speed up analysis
oxy_crit(squid.rd, thin = 1000)

## Experiment with different 'width' input
# Higher widths tend to oversmooth data
oxy_crit(squid.rd, width = 0.2)
# Lower width in this case gives very similar result to default 0.1
oxy_crit(squid.rd, width = 0.05)

## Run on oxygen~time data in 'inspect' object
insp <- inspect(squid.rd, time = 1, oxygen = 2)
oxy_crit(insp)

## Run on already calculated rate~oxygen data
# Calculate a rolling rate
rate <- auto_rate(squid.rd,
                  method = "rolling",
                  width = 0.1,
                  plot = FALSE)$rate

## Calculate a rolling mean oxygen
oxy <- na.omit(roll::roll_mean(squid.rd[[2]],
                               width = 0.1 * nrow(squid.rd)))
## Combine to data.frame
squid_rate_oxy <- data.frame(oxy, rate)
## Perform COV analysis
oxy_crit(squid_rate_oxy, oxygen = 1, rate = 2)
}
}
\references{
Yeager DP, Ultsch GR (1989) Physiological regulation and
conformation: A BASIC program for the determination of critical points.
Physiological Zoology 62:888–907. doi: 10.1086/physzool.62.4.30157935

Muggeo V (2008) Segmented: an R package to fit regression models with
broken-line relationships. R News 8:20–25.
}
