% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/postProcess.R
\name{cropInputs}
\alias{cropInputs}
\alias{cropInputs.default}
\alias{cropInputs.spatialClasses}
\alias{cropInputs.sf}
\title{Crop a \code{Spatial*} or \code{Raster*} object}
\usage{
cropInputs(
  x,
  studyArea,
  rasterToMatch,
  verbose = getOption("reproducible.verbose", 1),
  ...
)

\method{cropInputs}{default}(x, studyArea, rasterToMatch, ...)

\method{cropInputs}{spatialClasses}(
  x,
  studyArea = NULL,
  rasterToMatch = NULL,
  verbose = getOption("reproducible.verbose", 1),
  extentToMatch = NULL,
  extentCRS = NULL,
  useGDAL = getOption("reproducible.useGDAL", TRUE),
  useCache = getOption("reproducible.useCache", FALSE),
  ...
)

\method{cropInputs}{sf}(
  x,
  studyArea = NULL,
  rasterToMatch = NULL,
  verbose = getOption("reproducible.verbose", 1),
  extentToMatch = NULL,
  extentCRS = NULL,
  useCache = getOption("reproducible.useCache", FALSE),
  ...
)
}
\arguments{
\item{x}{A \code{Spatial*}, \code{sf}, or \code{Raster*} object.}

\item{studyArea}{\code{SpatialPolygons*} object used for masking and possibly cropping
if no \code{rasterToMatch} is provided.
If not in same CRS, then it will be \code{spTransform}ed to
CRS of \code{x} before masking. Currently, this function will not reproject the
\code{x}. Optional in \code{postProcess}.}

\item{rasterToMatch}{Template \code{Raster*} object used for cropping (so extent should be
the extent of desired outcome) and reprojecting (including changing the
resolution and projection).
See details in \code{\link{postProcess}}.}

\item{verbose}{Numeric, -1 silent (where possible), 0 being very quiet,
1 showing more messaging, 2 being more messaging, etc.
Default is 1. Above 3 will output much more information about the internals of
Caching, which may help diagnose Caching challenges. Can set globally with an
option, e.g., \code{options('reproducible.verbose' = 0) to reduce to minimal}}

\item{...}{Passed to \code{raster::crop}}

\item{extentToMatch}{Optional. Can pass an extent here and a \code{crs} to
\code{extentCRS} instead of \code{rasterToMatch}. These
will override \code{rasterToMatch}, with a warning if both
passed.}

\item{extentCRS}{Optional. Can pass a \code{crs} here with an extent to
\code{extentTomatch} instead of \code{rasterToMatch}}

\item{useGDAL}{Logical or \code{"force"}.
Defaults to \code{getOption("reproducible.useGDAL" = TRUE)}.
If \code{TRUE}, then this function will use \code{gdalwarp} only when not
small enough to fit in memory (i.e., \emph{if the operation fails} the
\code{raster::canProcessInMemory(x, 3)} test). Using \code{gdalwarp} will
usually be faster than \code{raster::projectRaster}, the function used
if this is \code{FALSE}. Since since the two options use different algorithms,
there may be different projection results. \code{"force"} will cause it to
use GDAL regardless of the memory test described here.}

\item{useCache}{Logical, default \code{getOption("reproducible.useCache", FALSE)}, whether
\code{Cache} is used internally.}
}
\description{
This function can be used to crop or reproject module inputs from raw data.
}
\examples{
# Add a study area to Crop and Mask to
# Create a "study area"
library(sp)
library(raster)
ow <- setwd(tempdir())

# make a SpatialPolygon
coords1 <- structure(c(-123.98, -117.1, -80.2, -100, -123.98, 60.9, 67.73, 65.58, 51.79, 60.9),
                     .Dim = c(5L, 2L))
Sr1 <- Polygon(coords1)
Srs1 <- Polygons(list(Sr1), "s1")
shpEcozone <- SpatialPolygons(list(Srs1), 1L)
crs(shpEcozone) <- "+proj=longlat +datum=WGS84 +no_defs +ellps=WGS84 +towgs84=0,0,0"

# make a "study area" that is subset of larger dataset
coords <- structure(c(-118.98, -116.1, -99.2, -106, -118.98, 59.9, 65.73, 63.58, 54.79, 59.9),
                    .Dim = c(5L, 2L))
Sr1 <- Polygon(coords)
Srs1 <- Polygons(list(Sr1), "s1")
StudyArea <- SpatialPolygons(list(Srs1), 1L)
crs(StudyArea) <- crs(shpEcozone)
projString <- "+proj=utm +zone=15 +ellps=GRS80 +datum=NAD83 +units=m +no_defs"
StudyArea <- sp::spTransform(StudyArea, CRSobj = projString)

##########
shpEcozonePostProcessed <- postProcess(shpEcozone, studyArea = StudyArea)


# Try manually, individual pieces
shpEcozoneReprojected <- projectInputs(shpEcozone, StudyArea)
shpEcozoneCropped <- cropInputs(shpEcozone, StudyArea)
shpEcozoneClean <- fixErrors(shpEcozone)
shpEcozoneMasked <- maskInputs(shpEcozone, StudyArea)

# With terra
if (require("terra")) {
  opts <- options("reproducible.useTerra" = TRUE)
  vectEcozone <- terra::vect(sf::st_as_sf(shpEcozone)) # direct conversion throws warning

  # If input is Spatial object --> return will also be Spatial
  shpEcozonePostProcessed <- postProcess(shpEcozone, studyArea = StudyArea)
  # Try manually, individual pieces -- Note functions are different
  shpEcozoneReprojected <- projectInputs(shpEcozone, StudyArea)
  shpEcozoneMasked <- maskInputs(shpEcozone, StudyArea)
  shpEcozoneCropped <- cropInputs(shpEcozone, StudyArea)

  # If input is Spat object --> return will also be Spat
  vectEcozonePostProcessed <- postProcess(vectEcozone, studyArea = StudyArea)
  # Try manually, individual pieces -- Note functions are different
  vectEcozoneMasked <- maskInputs(vectEcozone, StudyArea)
  VectEcozoneReprojected <- projectInputs(vectEcozone, StudyArea)
  vectEcozoneCropped <- cropInputs(vectEcozone, StudyArea)



  # Note these two have different function names --> methods for cropInputs and fixErrors
  #    are not implemented yet
  shpEcozoneClean <- fixErrorsTerra(vectEcozone)

  options(opts)
}

setwd(ow)
}
\author{
Eliot McIntire, Jean Marchal, Ian Eddy, and Tati Micheletti
}
