#'
#'
#' The main function that ties together and runs the models.
#' @param model lavaan output object.
#' @param n.lambda number of penalization values to test.
#' @param mult.start Logical. Whether to use multi_optim() (TRUE) or
#'         regsem() (FALSE).
#' @param multi.iter maximum number of random starts for multi_optim
#' @param jump Amount to increase penalization each iteration.
#' @param lambda.start What value to start the penalty at
#' @param type Penalty type. Options include "none", "lasso", "ridge",
#'        "enet" for the elastic net,
#'        "alasso" for the adaptive lasso, "scad, "mcp",
#'        and "diff_lasso". diff_lasso penalizes the discrepency between
#'        parameter estimates and some pre-specified values. The values
#'        to take the deviation from are specified in diff_par.
#' @param fit.ret Fit indices to return.
#' @param fit.ret2 Return fits using only dataset "train" or bootstrap "boot"? Have to
#'        do 2 sample CV manually.
#' @param data Optional dataframe. Only required for missing="fiml".
#' @param optMethod solver to use.
#' @param gradFun gradient function to use.
#' @param hessFun hessian function to use.
#' @param test.cov Covariance matrix from test dataset. Necessary for CV=T
#' @param parallel Logical. whether to parallelize the processes running models for all
#'        values of lambda.
#' @param ncore Number of cores to use when parallel=TRUE
#' @param Start type of starting values to use.
#' @param subOpt type of optimization to use in the optimx package.
#' @param pars_pen parameter indicators to penalize.
#' @param diff_par parameter values to deviate from.
#' @param LB lower bound vector.
#' @param UB upper bound vector
#' @param block Whether to use block coordinate descent
#' @param full Whether to do full gradient descent or block
#' @param calc Type of calc function to use with means or not. Not recommended
#'        for use.
#' @param nlminb.control list of control values to pass to nlminb
#' @param max.iter Number of iterations for coordinate descent
#' @param tol Tolerance for coordinate descent
#' @param solver Whether to use solver for coord_desc
#' @param solver.maxit Max iterations for solver in coord_desc
#' @param alpha.inc Whether alpha should increase for coord_desc
#' @param step Step size
#' @param momentum Logical for coord_desc
#' @param step.ratio Ratio of step size between A and S. Logical
#' @param warm.start Whether start values are based on previous iteration.
#'        This is not recommended.
#' @param missing How to handle missing data. Current options are "listwise"
#'        and "fiml".
#' @param ... Any additional arguments to pass to regsem() or multi_optim().
#' @keywords optim calc
#' @export
#' @examples
#' \dontrun{
#' library(lavaan)
#' HS <- data.frame(scale(HolzingerSwineford1939[,7:15]))
#' mod <- '
#' f =~ x1 + x2 + x3 + x4 + x5 + x6 + x7 + x8 + x9
#' '
#' outt = cfa(mod,HS)
#'
#' cv.out = cv_regsem(outt,type="ridge",gradFun="none",n.lambda=100)
#'}



cv_regsem = function(model,
                     n.lambda=100,
                     mult.start=TRUE,
                     multi.iter=100,
                     jump=0.002,
                     lambda.start=0,
                     type="none",
                     fit.ret=c("rmsea","BIC"),
                     fit.ret2 = "train",
                     data=NULL,
                     optMethod="default",
                    gradFun="ram",
                    hessFun="none",
                    test.cov=NULL,
                    parallel=FALSE,
                    ncore=2,
                    Start="lavaan",
                    subOpt="nlminb",
                    pars_pen=NULL,
                    diff_par=NULL,
                    LB=-Inf,
                    UB=Inf,
                    block=TRUE,
                    full=TRUE,
                    calc="normal",
                    max.iter=2000,
                    tol=1e-5,
                    solver=FALSE,
                    solver.maxit=5,
                    alpha.inc=TRUE,
                    step=.5,
                    momentum=FALSE,
                    step.ratio=FALSE,
                    nlminb.control=list(),
                    warm.start=TRUE,
                    missing="listwise",
                    ...){


#if(fit.ret2 == "test"){
#  ids <-  sample(nrow(dat),nrow(dat)/2)
#  dat.train <- dat[ids,]
#  dat.test <- dat[-ids,]
#}



if(parallel == TRUE){
  stop("parallel is not currently supported")
}

if(parallel==FALSE){
par.matrix <- matrix(0,n.lambda,length(extractMatrices(model)$parameters))
fits <- matrix(NA,n.lambda,length(fit.ret)+2)
SHRINK2 = lambda.start
count = 0
counts=n.lambda
#res2 <- data.frame(matrix(NA,counts,3))
#coefs = rep(1,14)

while(count < counts){
  count = count + 1
  print(count)
  SHRINK <- SHRINK2 + jump*(count-1) # 0.01 works well & 0.007 as well with 150 iterations

if(mult.start==FALSE){

  if(warm.start==FALSE | count == 1){
    itt = 0
    Start="lavaan"
  }else if(fits[count-1,2] == 0){
    itt = 0
    Start = par.matrix[count-1,]
    Start[pars_pen] = Start[pars_pen]-jump
  }else{
    itt = itt + 1
    Start = par.matrix[count-itt-1,]
    Start[pars_pen] = Start[pars_pen]-itt*jump
  }


  out <- regsem(model=model,lambda=SHRINK,type=type,data=data,
                   optMethod=optMethod,
                   gradFun=gradFun,hessFun=hessFun,
                   parallel=parallel,Start=Start,
                   subOpt=subOpt,
                   pars_pen=pars_pen,
                   diff_par=diff_par,
                   LB=LB,
                   UB=UB,
                   block=block,
                   full=full,
                   calc=calc,
                   tol=tol,
                    solver=solver,
                  solver.maxit=solver.maxit,
                  alpha.inc=alpha.inc,
                  step=step,
                  momentum=momentum,
                  step.ratio=step.ratio,
                   nlminb.control=nlminb.control,
                   missing=missing)


  }else if(mult.start==TRUE){

    if(warm.start==FALSE | count == 1){
      itt = 0
      Start2=NULL
    }else if(fits[count-1,2] == 0){
      itt = 0
      Start2 = par.matrix[count-1,]
      Start2[pars_pen] = Start2[pars_pen]-jump
    }else{
      itt = itt + 1
      Start2 = par.matrix[count-itt-1,]
      Start2[pars_pen] = Start2[pars_pen]-itt*jump
    }
   out <- multi_optim(model=model,max.try=multi.iter,lambda=SHRINK,
                      LB=LB,UB=UB,type=type,optMethod=optMethod,
                      gradFun=gradFun,hessFun=hessFun,
                      tol=tol,
                      solver=solver,
                      solver.maxit=solver.maxit,
                      full=full,
                      block=block,
                      alpha.inc=alpha.inc,
                      step=step,Start2=Start2,
                      momentum=momentum,
                      step.ratio=step.ratio,nlminb.control=nlminb.control,
                      pars_pen=pars_pen,diff_par=NULL)

  }
  #print(pars_pen)
 # pars_pen <- out$pars_pen
  #if(any(fit.ret2 == "test")==TRUE){
  #  fits[[count]]$test = NA #fit_indices(out,CV=TRUE)[fit.ret]
  #}else
  if(fit.ret2 == "train"){
    fitt = try(fit_indices(out,CV=FALSE)$fits[fit.ret],silent=T)
    if(inherits(fitt, "try-error")) {
      fits[count,3:ncol(fits)] = rep(NA,ncol(fits)-2)
    }else{
      fits[count,3:ncol(fits)] = fitt
    }

  }else if(fit.ret2 == "test"){
   # stop("fit.ret2=test is currently not implemented")
    #print(summary(out))
    fitt = try(fit_indices(out,CovMat=test.cov,CV=TRUE)$fits[fit.ret],silent=T)
    if(inherits(fitt, "try-error")) {
      fits[count,3:ncol(fits)] = rep(NA,ncol(fits)-2)
    }else{
      fits[count,3:ncol(fits)] = fitt
    }
  }else if(fit.ret2 == "boot"){
    fitt = try(fit_indices(out,CV="boot")$fits[fit.ret],silent=T)
    if(inherits(fitt, "try-error")) {
      fits[count,3:ncol(fits)] = rep(NA,ncol(fits)-2)
    }else{
      fits[count,3:ncol(fits)] = fitt
    }
  }
  fits[count,1] <- SHRINK

#  if(class(out$convergence)=="numeric"){
  #print(class(out$convergence));print(1)
  #print(out$convergence);print(class(out$convergence))
    fits[count,2] <- out$convergence

 # }else{
 #   fits[count,2] <- 99
    #out$convergence <- 99
 # }


  if(is.null(out$coefficients)==TRUE){
    break
  }
  par.matrix[count,] = as.matrix(out$coefficients)

  colnames(par.matrix) = names(out$coefficients)
  colnames(fits) <- c("lambda","conv",fit.ret)
  out2 <- list(par.matrix,fits,pars_pen)
 # ret

}
}else if(parallel==TRUE){



  par.matrix <- matrix(0,n.lambda,model@Fit@npar)
  fits <- matrix(NA,n.lambda,length(fit.ret)+2)
  SHRINK = 0
  count = 0
  counts=n.lambda
  #res2 <- data.frame(matrix(NA,counts,3))
  #coefs = rep(1,14)

  #library(snowfall)

  cv_parallel <- function(SHRINK){

    if(mult.start==FALSE){
      out <- regsem(model=model,lambda=SHRINK,type=type,data=data,
                    optMethod=optMethod,
                    gradFun=gradFun,hessFun=hessFun,
                    parallel=parallel,Start=Start,
                    subOpt=subOpt,
                    pars_pen=pars_pen,
                    diff_par=diff_par,
                    LB=LB,
                    UB=UB,
                    calc=calc,
                    nlminb.control=nlminb.control,
                    tol=tol,
                    full=full,
                    block=block,
                    solver=solver,
                    solver.maxit=solver.maxit,
                    alpha.inc=alpha.inc,
                    step=step,
                    momentum=momentum,
                    step.ratio=step.ratio,
                    missing=missing)


    }else if(mult.start==TRUE){
      out <- multi_optim(model=model,max.try=multi.iter,lambda=SHRINK,
                         LB=LB,UB=UB,type=type,optMethod=optMethod,
                         gradFun=gradFun,hessFun=hessFun,nlminb.control=nlminb.control,
                         tol=tol,
                         full=full,
                         block=block,
                         solver=solver,
                         solver.maxit=solver.maxit,
                         alpha.inc=alpha.inc,
                         step=step,
                         momentum=momentum,
                         step.ratio=step.ratio,
                         pars_pen=pars_pen,diff_par=NULL,warm.start=warm.start)
    }


    #if(any(fit.ret2 == "test")==TRUE){
    #  fits[[count]]$test = NA #fit_indices(out,CV=TRUE)[fit.ret]
    #}else
    if(fit.ret2 == "train"){
      fitt = try(fit_indices(out,CV=FALSE)$fits[fit.ret],silent=T)
      if(inherits(fitt, "try-error")) {
        fitss = rep(NA,ncol(fits)-2)
      }else{
        fitss = fitt
      }

    }else if(fit.ret2 == "test"){
      # stop("fit.ret2=test is currently not implemented")
      fitt = try(fit_indices(out,CovMat=test.cov,CV=TRUE)$fits[fit.ret],silent=T)
      if(inherits(fitt, "try-error")) {
        fitss = rep(NA,ncol(fits)-2)
      }else{
        fitss = fitt
      }
    }else if(fit.ret2 == "boot"){
      fitt = try(fit_indices(out,CV="boot")$fits[fit.ret],silent=T)
      if(inherits(fitt, "try-error")) {
        fitss = rep(NA,ncol(fits)-2)
      }else{
        fitss = fitt
      }
    }
    fitss <- matrix(fitss,1,length(fit.ret))
    data.frame(SHRINK,conv=out$convergence,fitss,out$coefficients)
  }



  snowfall::sfLibrary(regsem)
  snowfall::sfInit(parallel=TRUE, cpus=ncore)
  snowfall::sfExport("model","type","data",
                     "optMethod",
                     "gradFun","hessFun",
                     "parallel","Start",
                     "subOpt",
                     "pars_pen",
                     "diff_par",
                     "LB",
                     "block",
                     "full",
                     "UB",
                     "calc",
                     "nlminb.control",
                     "warm.start",
                     "missing")




  lambdas <- seq(0,by=jump,length.out=n.lambda)
  ret = snowfall::sfLapply(lambdas,cv_parallel)
  snowfall::sfStop()

  #out
  pars_pen <- out$pars_pen

  out2 <- unlist(ret)
  out2 <- matrix(out,nrow=n.lambda,ncol=length(ret[[1]]),byrow=T)
  nam <- names(extractMatrices(model)$parameters)
  colnames(out2) <- c("lambda","conv",fit.ret,nam)
  out2



}
#fits = fit_indices(out,CV=FALSE)
out2$pars_pen <- pars_pen
out2

}
