% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shiny.R
\name{updateAmBarChart}
\alias{updateAmBarChart}
\title{Update the data of a bar chart}
\usage{
updateAmBarChart(session, outputId, data)
}
\arguments{
\item{session}{the Shiny \code{session} object}

\item{outputId}{the output id passed on to \code{\link{amChart4Output}}}

\item{data}{new data; if it is not valid, then nothing will happen (in order
to be valid it must have the same structure as the data passed on to
\code{\link{amBarChart}} / \code{\link{amHorizontalBarChart}} /
\code{\link{amRadialBarChart}} / \code{\link{amStackedBarChart}}); in this
case check the JavaScript console, it will report the encountered issue}
}
\description{
Update the data of a bar chart in a Shiny app (vertical,
  horizontal, radial, or stacked bar chart).
}
\examples{
library(rAmCharts4)
library(shiny)

ui <- fluidPage(
  br(),
  actionButton("update", "Update", class = "btn-primary"),
  br(), br(),
  amChart4Output("barchart", width = "650px", height = "470px")
)

server <- function(input, output, session){

  set.seed(666)
  dat <- data.frame(
    country = c("USA", "China", "Japan", "Germany", "UK", "France"),
    visits = c(3025, 1882, 1809, 1322, 1122, 1114),
    income = rpois(6, 25),
    expenses = rpois(6, 20)
  )
  newdat <- data.frame(
    country = c("USA", "China", "Japan", "Germany", "UK", "France"),
    income = rpois(6, 25),
    expenses = rpois(6, 20)
  )

  output[["barchart"]] <- renderAmChart4({
    amBarChart(
      data = dat,
      category = "country",
      values = c("income", "expenses"),
      valueNames = list(income = "Income", expenses = "Expenses"),
      draggable = TRUE,
      backgroundColor = "#30303d",
      columnStyle = list(
        income = amColumn(
          color = "darkmagenta", strokeColor = "#cccccc", strokeWidth = 2
        ),
        expenses = amColumn(
          color = "darkred", strokeColor = "#cccccc", strokeWidth = 2
        )
      ),
      chartTitle = list(text = "Income and expenses per country"),
      xAxis = "Country",
      yAxis = "Income and expenses",
      yLimits = c(0, 41),
      valueFormatter = "#.#",
      caption = "Year 2018",
      theme = "dark")
  })

  observeEvent(input[["update"]], {
    updateAmBarChart(session, "barchart", newdat)
  })

}

if(interactive()){
  shinyApp(ui, server)
}


# Survival probabilities ####
library(shiny)
library(rAmCharts4)

probs <- c(control = 30, treatment = 75) # initial probabilities

ui <- fluidPage(
  br(),
  sidebarLayout(
    sidebarPanel(
      wellPanel(
        tags$fieldset(
          tags$legend("Survival probability"),
          sliderInput(
            "control",
            "Control group",
            min = 0, max = 100, value = probs[["control"]], step = 1
          ),
          sliderInput(
            "treatment",
            "Treatment group",
            min = 0, max = 100, value = probs[["treatment"]], step = 1
          )
        )
      )
    ),
    mainPanel(
      amChart4Output("barchart", width = "500px", height = "400px")
    )
  )
)

server <- function(input, output, session){

  dat <- data.frame(
    group = c("Control", "Treatment"),
    alive = c(probs[["control"]], probs[["treatment"]]),
    dead  = 100 - c(probs[["control"]], probs[["treatment"]])
  )
  stacks <- list(
    c("alive", "dead")
  )
  seriesNames <- list(
    alive = "Alive",
    dead  = "Dead"
  )

  output[["barchart"]] <- renderAmChart4({
    amStackedBarChart(
      dat,
      category = "group",
      stacks = stacks,
      seriesNames = seriesNames,
      yLimits = c(0, 100),
      chartTitle = amText(
        "Survival probabilities",
        fontFamily = "Trebuchet MS",
        fontSize = 30,
        fontWeight = "bold"
      ),
      xAxis = "Group",
      yAxis = "Probability",
      theme = "dataviz"
    )
  })

  observeEvent(list(input[["control"]], input[["treatment"]]), {
    newdat <- data.frame(
      group = c("Control", "Treatment"),
      alive = c(input[["control"]], input[["treatment"]]),
      dead  = 100 - c(input[["control"]], input[["treatment"]])
    )
    updateAmBarChart(session, "barchart", newdat)
  })

}

if(interactive()){
  shinyApp(ui, server)
}
}
