# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Partially Separable Function Optimization
#'
#' @description
#' Optimization method for specially structured partially separable
#' functions.
#'
#' @param par Initial values for the parameters. It is a concatenated
#' vector of the global parameters and all the private parameters.
#' @param fn Function to compute the element functions and their
#' derivatives. Each call computes on element function. See the examples
#' section.
#' @param n_ele_func Number of element functions.
#' @param rel_eps Relative convergence threshold.
#' @param n_threads Number of threads to use.
#' @param max_it Maximum number of iterations.
#' @param c1,c2 Thresholds for the Wolfe condition.
#' @param use_bfgs Logical for whether to use BFGS updates or SR1 updates.
#' @param trace Integer where larger values gives more information during the
#' optimization.
#' @param cg_tol Threshold for the conjugate gradient method.
#' @param strong_wolfe \code{TRUE} if the strong Wolfe condition should be used.
#' @param env Environment to evaluate \code{fn} in. \code{NULL} yields the
#' global environment.
#' @param max_cg maximum number of conjugate gradient iterations in each
#' iteration. Use zero if there should not be a limit.
#' @param pre_method preconditioning method in conjugate gradient method.
#' zero yields no preconditioning and one yields diagonal preconditioning.
#'
#' @details
#' The function follows the method described by Nocedal and Wright (2006)
#' and mainly what is described in Section 7.4. Details are provided
#' in the psqn vignette. See \code{vignette("psqn", package = "psqn")}.
#'
#' The partially separable function we consider are special in that the
#' function to be minimized is a sum of so-called element functions which
#' only depend on few shared (global) parameters and some
#' private parameters which are particular to each element function.
#'
#' The optimization function is also available in C++ as a header-only
#' library. Using C++ may reduce the computation time substantially. See
#' the vignette in the package for examples.
#'
#' @return
#' An object with the following elements:
#' \item{par}{the estimated global and private parameters.}
#' \item{value}{function value at \code{par}.}
#' \item{info}{information code. 0 implies convergence.
#' -1 implies that the maximum number iterations is reached.
#' -2 implies that the conjugate gradient method failed.
#' -3 implies that the line search failed.
#' -4 implies that the user interrupted the optimization.}
#' \item{counts}{An integer vector with the number of function evaluations,
#' gradient evaluations, and the number of conjugate gradient iterations.}
#' \item{convergence}{\code{TRUE} if \code{info == 0}.}
#'
#' @references
#' Nocedal, J. and Wright, S. J. (2006). \emph{Numerical Optimization}
#' (2nd ed.). Springer.
#'
#' @examples
#' # example with inner problem in a Taylor approximation for a mixed GLMM as
#' # in the vignette
#'
#' # assign model parameters, number of random effects, and fixed effects
#' q <- 2 # number of private parameters per cluster
#' p <- 1 # number of global parameters
#' beta <- sqrt((1:p) / sum(1:p))
#' Sigma <- diag(q)
#'
#' # simulate a data set
#' set.seed(66608927)
#' n_clusters <- 20L # number of clusters
#' sim_dat <- replicate(n_clusters, {
#'   n_members <- sample.int(8L, 1L) + 2L
#'   X <- matrix(runif(p * n_members, -sqrt(6 / 2), sqrt(6 / 2)),
#'               p)
#'   u <- drop(rnorm(q) %*% chol(Sigma))
#'   Z <- matrix(runif(q * n_members, -sqrt(6 / 2 / q), sqrt(6 / 2 / q)),
#'               q)
#'   eta <- drop(beta %*% X + u %*% Z)
#'   y <- as.numeric((1 + exp(-eta))^(-1) > runif(n_members))
#'
#'   list(X = X, Z = Z, y = y, u = u, Sigma_inv = solve(Sigma))
#' }, simplify = FALSE)
#'
#' # evalutes the negative log integrand.
#' #
#' # Args:
#' #   i cluster/element function index.
#' #   par the global and private parameter for this cluster. It has length
#' #       zero if the number of parameters is requested. That is, a 2D integer
#' #       vector the number of global parameters as the first element and the
#' #       number of private parameters as the second element.
#' #   comp_grad logical for whether to compute the gradient.
#' r_func <- function(i, par, comp_grad){
#'   dat <- sim_dat[[i]]
#'   X <- dat$X
#'   Z <- dat$Z
#'
#'   if(length(par) < 1)
#'     # requested the dimension of the parameter
#'     return(c(global_dim = NROW(dat$X), private_dim = NROW(dat$Z)))
#'
#'   y <- dat$y
#'   Sigma_inv <- dat$Sigma_inv
#'
#'   beta <- par[1:p]
#'   uhat <- par[1:q + p]
#'   eta <- drop(beta %*% X + uhat %*% Z)
#'   exp_eta <- exp(eta)
#'
#'   out <- -sum(y * eta) + sum(log(1 + exp_eta)) +
#'     sum(uhat * (Sigma_inv %*% uhat)) / 2
#'   if(comp_grad){
#'     d_eta <- -y + exp_eta / (1 + exp_eta)
#'     grad <- c(X %*% d_eta,
#'               Z %*% d_eta + dat$Sigma_inv %*% uhat)
#'     attr(out, "grad") <- grad
#'   }
#'
#'   out
#' }
#'
#' # optimize the log integrand
#' res <- psqn(par = rep(0, p + q * n_clusters), fn = r_func,
#'             n_ele_func = n_clusters)
#' head(res$par, p)              # the estimated global parameters
#' tail(res$par, n_clusters * q) # the estimated private parameters
#'
#' # compare with
#' beta
#' c(sapply(sim_dat, "[[", "u"))
#' @export
psqn <- function(par, fn, n_ele_func, rel_eps = .00000001, max_it = 100L, n_threads = 1L, c1 = .0001, c2 = .9, use_bfgs = TRUE, trace = 0L, cg_tol = .5, strong_wolfe = TRUE, env = NULL, max_cg = 0L, pre_method = 1L) {
    .Call(`_psqn_psqn`, par, fn, n_ele_func, rel_eps, max_it, n_threads, c1, c2, use_bfgs, trace, cg_tol, strong_wolfe, env, max_cg, pre_method)
}

#' BFGS Implementation Used Internally in the psqn Package
#'
#' @description
#' The method seems to mainly differ from \code{\link{optim}} by the line search
#' method. This version uses the interpolation method with a zoom phase
#' using cubic interpolation as described by Nocedal and Wright (2006).
#'
#' @references
#' Nocedal, J. and Wright, S. J. (2006). \emph{Numerical Optimization}
#' (2nd ed.). Springer.
#'
#' @return
#' An object like the object returned by \code{\link{psqn}}.
#'
#' @inheritParams psqn
#' @param par Initial values for the parameters.
#' @param fn Function to evaluate the function to be minimized.
#' @param gr Gradient of \code{fn}. Should return the function value as an
#' attribute called \code{"value"}.
#' @param env Environment to evaluate \code{fn} and \code{gr} in.
#' \code{NULL} yields the global environment.
#' @export
#'
#' @examples
#' # declare function and gradient from the example from help(optim)
#' fn <- function(x) {
#'   x1 <- x[1]
#'   x2 <- x[2]
#'   100 * (x2 - x1 * x1)^2 + (1 - x1)^2
#' }
#' gr <- function(x) {
#'   x1 <- x[1]
#'   x2 <- x[2]
#'   c(-400 * x1 * (x2 - x1 * x1) - 2 * (1 - x1),
#'      200 *      (x2 - x1 * x1))
#' }
#'
#' # we need a different function for the method in this package
#' gr_psqn <- function(x) {
#'   x1 <- x[1]
#'   x2 <- x[2]
#'   out <- c(-400 * x1 * (x2 - x1 * x1) - 2 * (1 - x1),
#'             200 *      (x2 - x1 * x1))
#'   attr(out, "value") <- 100 * (x2 - x1 * x1)^2 + (1 - x1)^2
#'   out
#' }
#'
#' # we get the same
#' optim    (c(-1.2, 1), fn, gr, method = "BFGS")
#' psqn_bfgs(c(-1.2, 1), fn, gr_psqn)
#'
#' # compare the computation time
#' system.time(replicate(1000,
#'                       optim    (c(-1.2, 1), fn, gr, method = "BFGS")))
#' system.time(replicate(1000,
#'                       psqn_bfgs(c(-1.2, 1), fn, gr_psqn)))
psqn_bfgs <- function(par, fn, gr, rel_eps = .00000001, max_it = 100L, c1 = .0001, c2 = .9, trace = 0L, env = NULL) {
    .Call(`_psqn_psqn_bfgs`, par, fn, gr, rel_eps, max_it, c1, c2, trace, env)
}

