% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/func_riedsid.R
\name{riedsid}
\alias{riedsid}
\alias{riedsid.spec}
\alias{riedsid.default}
\alias{riedsid2}
\alias{riedsid2.spec}
\alias{riedsid2.default}
\title{Constrained, optimal tapers using the Riedel & Sidorenko--Parker method}
\usage{
riedsid(PSD, ...)

\method{riedsid}{spec}(PSD, ...)

\method{riedsid}{default}(PSD, ntaper = 1L, tapseq = NULL,
  Deriv.method = c("local_qls", "spg"), constrained = TRUE,
  c.method = NULL, verbose = TRUE, ...)

riedsid2(PSD, ...)

\method{riedsid2}{spec}(PSD, ...)

\method{riedsid2}{default}(PSD, ntaper = 1L, constrained = TRUE,
  verbose = TRUE, fast = FALSE, ...)
}
\arguments{
\item{PSD}{vector or class \code{'amt'} or \code{'spec'}; the spectral values used to optimize taper numbers}

\item{...}{optional arguments passed to \code{\link{constrain_tapers}}}

\item{ntaper}{scalar or vector; number of tapers to apply optimization}

\item{tapseq}{vector; representing positions or frequencies (same length as \code{PSD})}

\item{Deriv.method}{character; choice of gradient estimation method}

\item{constrained}{logical; apply constraints with \code{\link{constrain_tapers}}; \code{FALSE} turns off constraints}

\item{c.method}{string; constraint method to use with \code{\link{constrain_tapers}}, only if \code{constrained=TRUE}}

\item{verbose}{logical; should messages be printed?}

\item{fast}{logical; use faster method?}
}
\value{
Object with class \code{'tapers'}
}
\description{
Estimates the optimal number of tapers at each frequency of
given PSD, using a modified Riedel-Sidorenko MSE recipe (RS-RLP).
}
\details{
The optimization is as follows. First, weighted derivatives of the 
input PSD are computed.
Using those derivates the optimal number of tapers is found through the 
RS-RLP formulation.
Constraints are then placed on the practicable number of tapers.

\code{\link{riedsid2}} is a new (faster) implementation which does not allow 
for multiple constraint methods; this is the preferred function to use.

\subsection{Taper constraints}{
The parameter \code{c.method} provides an option to change the method
of taper constraints.  A description of each may be found in 
the documentation for \code{\link{constrain_tapers}}.

Once can use \code{constrained=FALSE} to turn off all taper constraints; this
could lead to strange behavior though.
}

\subsection{Spectral derivatives}{
The parameter \code{Deriv.method} determines which method is used
to estimate derivatives.
\itemize{
\item{\code{"local_qls"}}{ (\strong{default}) uses quadratic weighting and
local least-squares estimation; this can be slower than \code{"spg"}.}
\item{\code{"spg"}}{ uses \code{\link{splineGrad}}; then, additional arguments
may be passed to control the smoothness of the derivatives
(e.g \code{spar} in \code{\link{smooth.spline}}).}
}
}
}
\section{Warning}{

The \code{"spg"} can become numerically unstable, and it's not clear when it will
be the preferred over the \code{"local_qls"} method, other than for efficiency's sake.
}

\examples{
\dontrun{#REX
library(psd)

##
## Riedel-Sidorenko-Parker taper optimization
##

set.seed(1234)
# some params
nd <- 512 # num data
ntap <- 10 # num tapers
nrm <- 40 # sharpness of the peaks rel 2*variance
#
# create a pseudo spectrum
# with broad peaks
x <- 0:(nd-1)
riex <- rnorm(nd) + nrm*abs(cos(pi*x/180) + 1.2)
riex <- riex + 8*nrm*dcauchy(x, nd/3)
riex <- riex + 5*nrm*dnorm(x, nd/2)
# some flat regions
riex[riex<25] <- 25
ried <- dB(riex, invert=TRUE)

# optimize tapers
rtap <- riedsid(riex, ntaper=ntap) # deprecation warning
rtap2 <- riedsid2(riex, ntaper=ntap)
rtap3 <- riedsid2(riex, ntaper=ntap, fast=TRUE)

# plot
op <- par(no.readonly = TRUE)
par(mfrow=c(2,1), mar=rep(1.3,4), mai=rep(0.6,4))
# ... the mock spectrum
plot(riex, type="h", xaxs="i", ylim=c(0,200), 
     main='Pseudo-spectrum') 

# ... tapers
plot(rtap2, col=NA, xaxs="i",
     main='Original and Optimized tapers', 
     ylim=c(0,max(c(ntap, rtap,rtap2,rtap3)))) 
# original tapers:
abline(h=ntap, lty=2)
# optimized tapers
lines(rtap, col="red")
# 2 and 2-fast
lines(rtap2, lwd=3, col="blue")
lines(rtap3, col="cyan")
par(op)

}#REX
}
\seealso{
\code{\link{constrain_tapers}}, \code{\link{resample_fft_rcpp}}, \code{\link{psdcore}}, \code{\link{pspectrum}}
}
\author{
A.J. Barbour adapted original by R.L. Parker
}
