% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/profileCI.R
\name{profileCI}
\alias{profileCI}
\title{Confidence Intervals using Profile Likelihood}
\usage{
profileCI(
  object,
  loglik,
  ...,
  parm = "all",
  level = 0.95,
  profile = TRUE,
  mult = 32,
  faster = TRUE,
  epsilon = -1,
  optim_args = list()
)
}
\arguments{
\item{object}{A fitted model object. This object must have a \code{coef} S3
method. If \code{faster = TRUE} then it must also have a \code{vcov} S3 method.}

\item{loglik}{A named function that returns the log-likelihood based on
input parameter values and data. The first argument must be the vector of
model parameters. If the likelihood is zero for any observation in the
data then the function should return \code{-Inf.}

Alternatively, \code{loglik} does not need to be supplied if a \code{\link{logLikFn}} S3
method has been created for \code{object}. The \code{profileCI} package provides
\code{logLikFn.glm}, which is used in an example in \strong{Examples}.}

\item{...}{Further arguments to be passed to \code{loglik}.}

\item{parm}{A vector specifying the parameters for which confidence
intervals are calculated, either a vector of numbers or a vector of names.
The default, \code{which = "all"}, produces confidence intervals for all the
parameters.}

\item{level}{The confidence level required.  A numeric scalar in (0, 1).}

\item{profile}{A logical scalar. If \code{TRUE} then confidence intervals
based on a profile log-likelihood are returned.  If \code{FALSE} then intervals
based on approximate large sample normal theory, which are symmetric about
the MLE, are returned.}

\item{mult}{A positive numeric scalar. Controls the increment by which the
parameter of interest is increased/decreased when profiling above/below
its MLE. The increment is \code{mult * se / 100} where \code{se} is the estimated
standard error of the estimator of the parameter. Decreasing \code{mult}
profiles at more points but will be slower.}

\item{faster}{A logical scalar. If \code{faster = TRUE} then the profiling of the
log-likelihood is in search of a lower (upper) confidence limit is
started at the corresponding symmetric lower (upper) confidence limit.}

\item{epsilon}{Only relevant if \code{profile = TRUE}. A numeric vector of values
that determine the accuracy of the confidence limits. \code{epsilon} is
recycled to the length of the parameter vector \code{parm}.
\itemize{
\item If \code{epsilon[i] > 0} then this value is passed as the argument \code{epsilon}
to the \code{\link[itp:itp]{itp::itp}} function, which estimates the parameter values for
which the profile log-likelihood for parameter \code{i} drops to the value
that defines the confidence limits, once profiling has been successful
in finding an interval within which this value lies.
\item If \code{epsilon[i] < 0} quadratic interpolation is used, which will tend to
be faster.
\item If \code{epsilon[i] = 0} then linear interpolation is used, which will be
faster still.
}}

\item{optim_args}{A list of further arguments (other than \code{par} and \code{fn}) to
pass to \verb{[stats::optim]}.}
}
\value{
An object of class \code{c("profileCI", "matrix", "array")}. A numeric
matrix with 2 columns giving the lower and upper confidence limits for
each parameter. These columns are labelled as \code{(1-level)/2} and
\code{1-(1-level)/2}, expressed as a percentage, by default \verb{2.5\%} and \verb{97.5\%}.
The row names are the names of the parameters supplied in \code{parm}.

If \code{profile = TRUE} then the returned object has extra attributes \code{crit},
\code{level} and \code{for_plot}. The latter is a named list of length equal to the
number of parameters. Each component is a 2-column numeric matrix. The
first column contains values of the parameter and the second column the
corresponding values of the profile log-likelihood. The profile
log-likelihood is equal to the attribute \code{crit} at the limits of the
confidence interval. The attribute \code{level} is the input argument \code{level}.
If \code{faster = FALSE} or \code{epsilon > 0} then the attributes \code{lower_pars} and
\code{upper_pars} are lists that provide, for each profiling, the values of the
parameters for the last maximisation of the log-likelihood.

A matrix with columns giving the  object
\code{c("profileCI", "matrix", "array")}.
}
\description{
Calculates confidence intervals for one or more parameters in a fitted
model object. A function that returns the log-likelihood must be supplied,
either directly via the argument \code{loglik} or using a \code{\link{logLikFn}} S3
generic.
}
\details{
The default, \code{epsilon = -1}, should work well enough in most
circumstances, but to achieve a specific accuracy set \code{epsilon} to be
a small positive value, for example, \code{epsilon = 1e-4}.

The defaults \code{mult  = 32} and \code{faster = TRUE} are designed to calculate
confidence intervals fairly quickly. If the object returned from
\code{profileCI} is plotted, using \code{\link{plot.profileCI}}, then we will not obtain
a smooth plot of a profile log-likelihood. Setting \code{faster = FALSE} and
reducing \code{mult}, perhaps to \code{8} or \code{16} should produce a smoother plot.
}
\examples{
## From example(glm)
counts <- c(18, 17, 15, 20, 10, 20, 25, 13, 12)
outcome <- gl(3, 1, 9)
treatment <- gl(3, 3)
glm.D93 <- glm(counts ~ outcome + treatment, family = poisson())
confint(glm.D93)
confint.default(glm.D93)

# A logLikFn.glm S3 method is provided in profileCI so we do not need to
# supply loglik explicitly
prof <- profileCI(glm.D93)
prof

# Supplying a Poisson GLM log-likelihood explicitly
poisson_loglik <- function(pars) {
  lambda <- exp(model.matrix(glm.D93) \%*\% pars)
  loglik <- stats::dpois(x = glm.D93$y, lambda = lambda, log = TRUE)
  return(sum(loglik))
}
# This will be a bit slower than profile.glm() because glm.fit() is fast
prof <- profileCI(glm.D93, loglik = poisson_loglik)
prof
plot(prof, parm = 1)
plot(prof, parm = "outcome2")

# Supplying a more general Poisson GLM log-likelihood
poisson_loglik_2 <- function(pars, glm_object) {
  lambda <- exp(model.matrix(glm_object) \%*\% pars)
  loglik <- stats::dpois(x = glm_object$y, lambda = lambda, log = TRUE)
  return(sum(loglik))
}
prof <- profileCI(glm.D93, loglik = poisson_loglik_2, glm_object = glm.D93)
prof
}
\seealso{
\code{\link{plot.profileCI}} and \code{\link{print.profileCI}}.
}
