% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/processanimateR.R
\name{animate_process}
\alias{animate_process}
\title{Animate cases on a process map}
\usage{
animate_process(eventlog, processmap = process_map(eventlog, render = F,
  ...), animation_mode = "absolute", animation_duration = 60,
  animation_jitter = 0, animation_timeline = TRUE, token_size = NULL,
  token_color = NULL, token_image = NULL, token_opacity = NULL,
  token_shape = c("circle", "rect"), token_options = NULL,
  width = NULL, height = NULL, ...)
}
\arguments{
\item{eventlog}{The event log object that should be animated}

\item{processmap}{The process map created with processmapR that the event log should be animated on,
if not provided a standard process map will be generated by using processmapR::process_map.}

\item{animation_mode}{Whether to animate the cases according to their actual time of occurence ("absolute") or to start all cases at once ("relative").}

\item{animation_duration}{The overall duration of the animation, all times are scaled according to this overall duration.}

\item{animation_jitter}{The magnitude of a random coordinate translation, known as jitter in scatterplots, which is added to each token. Adding jitter can help to disambiguate tokens traveling on top of each other.}

\item{animation_timeline}{Whether to render a timeline slider in supported browsers (Recent versions of Chrome and Firefox only).}

\item{token_size}{The event attribute (character) or alternatively a data frame with three columns (case, time, size) matching the case identifier of the supplied event log.
The token size is scaled accordingly during the animation (default size is 4). You may use \code{\link{add_token_size}} to add a suitable attribute to the event log.}

\item{token_color}{The event attribute (character) or alternatively a data frame with three columns (case, time, color) matching the case identifier of the supplied event log.
The token color is change accordingly during the animation (default color is orange). You may use \code{\link{add_token_color}} to add a suitable attribute to the event log.}

\item{token_image}{The event attribute (character) or alternatively a data frame with three columns (case, time, image) matching the case identifier of the supplied event log.
The token image is change accordingly during the animation (by default a SVG shape is used).}

\item{token_opacity}{The event attribute (character) or alternatively a data frame with three columns (case, time, transparency) matching the case identifier of the supplied event log.
The token fill-opacity is change accordingly during the animation (by default the token is dranw with 0.9 opacity).}

\item{token_shape}{The (fixed) SVG shape to be used to draw tokens. Can be either 'circle' (default) or 'rect'.}

\item{token_options}{A list of additional (fixed) SVG properties to be added to each token.}

\item{width, height}{Fixed size for widget (in css units). The default is NULL, which results in intelligent automatic sizing based on the widget's container.}

\item{...}{Options passed on to \code{\link{process_map}}.}
}
\description{
A function for creating a SVG animation of an event log on a process map created by processmapR.
}
\examples{
# Load example event log
library(eventdataR)

# Animate the process with default options (absolute time and 60s duration)
animate_process(patients)

# Animate the process with default options (relative time and with jitter)
animate_process(patients, animation_mode = "relative", animation_jitter = 10)

\donttest{

#' # Change default token sizes
animate_process(patients, token_size = 2)

# Change default token color
animate_process(patients, token_color = "red")

# Change default token opacity
animate_process(patients, token_opacity = 0.5)

# Change default token image (GIFs work too)
animate_process(patients, token_image = "https://upload.wikimedia.org/wikipedia/en/5/5f/Pacman.gif")

# Change token color based on a numeric attribute, here the nonsensical 'time' of an event
animate_process(add_token_color(patients, "time", "color"), token_color = "color")

# Change token color based on a factor attribute
animate_process(add_token_color(patients, "employee", "color",
                color_mapping = scales::col_factor("Set3", patients$employee)),
                token_color = "color")

# Next example requires the 'dplyr' and 'edeaR' packages
library(dplyr)
library(edeaR)

# Change token_color based on colors in a second data frame
# Extract only the lacticacid measurements
lactic <- sepsis \%>\%
    mutate(lacticacid = as.numeric(lacticacid)) \%>\%
    filter_activity(c("LacticAcid")) \%>\%
    as.data.frame() \%>\%
    select("case" = case_id, "time" =  timestamp, lacticacid)
# Create a numeric color scale
cscale <- scales::col_numeric("Oranges", lactic$lacticacid , na.color = "white")
# Create colors data frame for animate_process
lacticColors <- lactic \%>\%
    mutate(color = cscale(lacticacid))
sepsisBase <- sepsis \%>\%
    filter_activity(c("LacticAcid", "CRP", "Leucocytes", "Return ER",
                      "IV Liquid", "IV Antibiotics"), reverse = TRUE) \%>\%
    filter_trace_frequency(percentage = 0.95)
animate_process(sepsisBase, token_color = lacticColors, animation_mode = "relative",
                animation_duration = 600)
}


}
\seealso{
processmapR:process_map
}
\author{
Felix Mannhardt <felix.mannhardt@sintef.no> (SINTEF Digital)
}
