#' @include internal.R Collection-proto.R
NULL

#' Add a penalty
#'
#' A penalty can be applied to a conservation planning [problem()] to
#' penalize solutions according to a specific metric. Penalties---unlike
#' [constraints]---act as an explicit trade-off with the objective
#' being minimized or maximized (e.g. solution cost when used with
#' [add_min_set_objective()]).
#'
#' @details Both penalties and constraints can be used to modify a problem and
#'   identify solutions that exhibit specific characteristics. Constraints work
#'   by invalidating solutions that do not exhibit specific characteristics.
#'   On the other hand, penalties work by specifying trade-offs against the
#'   main problem objective and are mediated by a penalty factor.
#'
#'   The following penalties can be added to a conservation planning
#'   [problem()]:
#'
#'   \describe{
#'
#'   \item{[add_boundary_penalties()]}{Add penalties to a
#'     conservation problem to favor solutions that have
#'     planning units clumped together into contiguous areas.}
#'
#'   \item{[add_connectivity_penalties()]}{Add penalties to a
#'     conservation problem to favor solutions that select
#'     planning units with high connectivity between them.}
#'
#'   \item{[add_linear_penalties()]}{Add penalties to a
#'     conservation problem to favor solutions that avoid selecting
#'     planning units based on a certain variable
#'     (e.g. anthropogenic pressure).}
#'
#'   }
#'
#' @seealso [constraints], [decisions],
#'  [objectives] [portfolios], [problem()],
#'  [solvers], [targets].
#'
#' @examples
#' # load data
#' data(sim_pu_raster, sim_features)
#'
#' # create basic problem
#' p1 <- problem(sim_pu_raster, sim_features) %>%
#'       add_min_set_objective() %>%
#'       add_relative_targets(0.2) %>%
#'       add_default_solver(verbose = FALSE)
#'
#' # create problem with boundary penalties
#' p2 <- p1 %>% add_boundary_penalties(5, 1)
#'
#' # create connectivity matrix based on spatial proximity
#'  scm <- as.data.frame(sim_pu_raster, xy = TRUE, na.rm = FALSE)
#'  scm <- 1 / (as.matrix(dist(scm)) + 1)
#'
#' # remove weak and moderate connections between planning units to reduce
#' # run time
#' scm[scm < 0.85] <- 0
#'
#' # create problem with connectivity penalties
#' p3 <- p1 %>% add_connectivity_penalties(25, data = scm)
#'
#' # create problem with linear penalties,
#' # here the penalties will be based on random numbers to keep it simple
#'
#' # simulate penalty data
#' sim_penalty_raster <- simulate_cost(sim_pu_raster)
#'
#' # plot penalty data
#' plot(sim_penalty_raster, main = "penalty data", axes = FALSE, box = FALSE)
#'
#' # create problem with linear penalties, with a penalty scaling factor of 100
#' p4 <- p1 %>% add_linear_penalties(100, data = sim_penalty_raster)
#'
#' \dontrun{
#' # solve problems
#' s <- stack(solve(p1), solve(p2), solve(p3), solve(p4))
#'
#' # plot solutions
#' plot(s, axes = FALSE, box = FALSE,
#'      main = c("basic solution", "boundary penalties",
#'               "connectivity penalties", "linear penalties"))
#'  }
#' @name penalties
NULL
