% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/step-select.R
\name{step.CR}
\alias{step.CR}
\title{Curtis--Reid automatic step selection}
\usage{
step.CR(
  FUN,
  x,
  h0 = 1e-05 * max(abs(x), sqrt(.Machine$double.eps)),
  max.rel.error = .Machine$double.eps^(7/8),
  version = c("original", "modified"),
  aim = if (version[1] == "original") 100 else 1,
  acc.order = c(2L, 4L),
  tol = if (version[1] == "original") 10 else 4,
  range = h0/c(1e+05, 1e-05),
  maxit = 20L,
  seq.tol = 1e-04,
  cores = 1,
  preschedule = getOption("pnd.preschedule", TRUE),
  cl = NULL,
  ...
)
}
\arguments{
\item{FUN}{Function for which the optimal numerical derivative step size is needed.}

\item{x}{Numeric scalar: the point at which the derivative is computed and the optimal step size is estimated.}

\item{h0}{Numeric scalar: initial step size, defaulting to a relative step of
slightly greater than .Machine$double.eps^(1/3) (or absolute step if \code{x == 0}).}

\item{max.rel.error}{Error bound for the relative function-evaluation error
(\eqn{\frac{\hat f(\hat x) - f(x)}{f(x)}}{(^f(^x) - f(x))/f(x)}). Measures how noisy a function is.
If the function is relying on numerical optimisation routines, consider setting to
\code{sqrt(.Machine$double.eps)}.
If the function has full precision to the last bit, set to \code{.Machine$double.eps/2}.}

\item{version}{Character scalar: \code{"original"} for the original 1974 version by
Curtis and Reid; \code{"modified"} for Kostyrka’s 2025 modification, which adds an
extra evaluation for a more accurate estimate of the truncation error.}

\item{aim}{Positive real scalar: desired ratio of truncation-to-rounding error. The \code{"original"}
version over-estimates the truncation error, hence a higher \code{aim} is recommended.
For the \code{"modified"} version, aim should be close to 1.}

\item{acc.order}{Numeric scalar: in the modified version, allows searching for a
step size that would be optimal for a 4th-order-accurate central difference
See the Details section below.}

\item{tol}{Numeric scalar greater than 1: tolerance multiplier for determining when to stop
the algorithm based on the current estimate being between \code{aim/tol} and \code{aim*tol}.}

\item{range}{Numeric vector of length 2 defining the valid search range for the step size.}

\item{maxit}{Integer: maximum number of algorithm iterations to prevent infinite
loops in degenerate cases.}

\item{seq.tol}{Numeric scalar: maximum relative difference between old and new
step sizes for declaring convergence.}

\item{cores}{Integer specifying the number of CPU cores used for parallel computation.
Recommended to be set to the number of physical cores on the machine minus one.}

\item{preschedule}{Logical: if \code{TRUE}, disables pre-scheduling for \code{mclapply()}
or enables load balancing with \code{parLapplyLB()}. Recommended for functions that
take less than 0.1 s per evaluation.}

\item{cl}{An optional user-supplied \code{cluster} object  (created by \code{makeCluster}
or similar functions). If not \code{NULL}, the code uses \code{parLapply()} (if \code{preschedule}
is \code{TRUE}) or \code{parLapplyLB()} on that cluster on Windows, and \code{mclapply}
(fork cluster) on everything else.}

\item{...}{Passed to \code{FUN}.}
}
\value{
A list similar to the one returned by \code{optim()}:
\itemize{
\item \code{par} – the optimal step size found.
\item \code{value} – the estimated numerical first derivative (using central differences;
especially useful for computationally expensive functions).
\item \code{counts} – the number of iterations (each iteration includes three function evaluations).
\item \code{abs.error} – an estimate of the truncation and rounding errors.
\item \code{exitcode} – an integer code indicating the termination status:
\itemize{
\item \code{0} – Optimal termination within tolerance.
\item \code{1} – Third derivative is zero; large step size preferred.
\item \code{2} – No change in step size within tolerance.
\item \code{3} – Solution lies at the boundary of the allowed value range.
\item \code{4} – Maximum number of iterations reached.
}
\item \code{message} – A summary message of the exit status.
\item \code{iterations} – A list including the full step size search path, argument grids,
function values on those grids, estimated error ratios, and estimated derivative values.
}
}
\description{
Curtis--Reid automatic step selection
}
\details{
This function computes the optimal step size for central differences using the
\insertCite{curtis1974choice}{pnd} algorithm.
If the estimated third derivative is exactly zero, then, the initial step size
is multiplied by 4 and returned.

If 4th-order accuracy (4OA) is requested, then, two things happen. Firstly,
since 4OA differences requires a larger step size and the truncation error for
the 2OA differences grows if the step size is larger than the optimal one,
a higher ratio of truncation-to-rounding errors should be targeted. Secondly,
a 4OA numerical derivative is returned, but the truncation and rounding errors
are still estimated for the 2OA differences. Therefore, the estimating truncation
error is higher and the real truncation error of 4OA differences is lower.

TODO: mention that f must be one-dimensional

The arguments passed to \code{...} must not partially match those of \code{\link[=step.CR]{step.CR()}}. For example, if
\code{cl} exists, then, attempting to avoid cluster export by using
\code{step.CR(f, x, h = 1e-4, cl = cl, a = a)} will result in an error: \code{a} matches \code{aim}
and \code{acc.order}. Redefine the function for this argument to have a name that is not equal
to the beginning of one of the arguments of \code{\link[=step.CR]{step.CR()}}.
}
\examples{
f <- function(x) x^4
step.CR(x = 2, f)
step.CR(x = 2, f, h0 = 1e-3)
step.CR(x = 2, f, version = "modified")
step.CR(x = 2, f, version = "modified", acc.order = 4)

# A bad start: too far away
step.CR(x = 2, f, h0 = 1000)  # Bad exit code + a suggestion to extend the range
step.CR(x = 2, f, h0 = 1000, range = c(1e-10, 1e5))  # Problem solved

library(parallel)
cl <- makePSOCKcluster(names = 2, outfile = "")
abc <- 2
f <- function(x, abc) {Sys.sleep(0.02); abc*sin(x)}
x <- pi/4
system.time(step.CR(f, x, h = 1e-4, cores = 1, abc = abc))  # To remove speed-ups
system.time(step.CR(f, x, h = 1e-4, cores = 2, abc = abc))  # Faster
f2 <- function(x) f(x, abc)
clusterExport(cl, c("f2", "f", "abc"))
system.time(step.CR(f2, x, h = 1e-4, cl = cl))  # Also fast
stopCluster(cl)
}
\references{
\insertAllCited{}
}
