% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitacis.R
\name{fitacis}
\alias{fitacis}
\alias{plot.acifits}
\title{Fit multiple A-Ci curves at once}
\usage{
fitacis(data, group, fitmethod = c("default", "bilinear"),
  progressbar = TRUE, quiet = FALSE, id = NULL, ...)

\method{plot}{acifits}(x, how = c("manyplots", "oneplot"), highlight = NULL,
  ylim = NULL, xlim = NULL, add = FALSE, what = c("model", "data",
  "none"), ...)
}
\arguments{
\item{data}{Dataframe with Ci, Photo, Tleaf, PPFD (the last two are optional). For \code{fitacis}, also requires a grouping variable.}

\item{group}{The name of the grouping variable in the dataframe (an A-Ci curve will be fit for each group separately).}

\item{fitmethod}{Method to fit the A-Ci curve. Either 'default' (Duursma 2015), or 'bilinear'. See Details.}

\item{progressbar}{Display a progress bar (default is TRUE).}

\item{quiet}{If TRUE, no messages are written to the screen.}

\item{id}{Names of variables (quoted, can be a vector) in the original dataset to return as part of the coef() statement. Useful for keeping track of species names, treatment levels, etc. See Details and Examples.}

\item{x}{For \code{plot.acifits}, an object returned from \code{fitacis}}

\item{how}{If 'manyplots', produces a single plot for each A-Ci curve. If 'oneplot' overlays all of them.}

\item{highlight}{If a name of a curve is given (check names(object), where object is returned by acifits), all curves are plotted in grey, with the highlighted one on top.}

\item{xlim, ylim}{The X and Y axis limits.}

\item{add}{If TRUE, adds the plots to a current plot.}

\item{what}{What to plot, either 'model' (the fitted curve), 'data' or 'none'. See examples.}

\item{\dots}{Further arguments passed to \code{\link{fitaci}} (in the case of \code{fitacis}), or \code{\link{plot.acifit}} (in the case of \code{plot.acifits}).}
}
\description{
A convenient function to fit many curves at once, by calling \code{\link{fitaci}} for every group in the dataset. The data provided must include a variable that uniquely identifies each A-Ci curve.
}
\details{
\strong{Troubleshooting - } When using the default fitting method (see \code{\link{fitaci}}), it is common that some curves cannot be fit. Usually this indicates that the curve is poor quality and should not be used to estimate photosynthetic capacity, but there are exceptions. The \code{fitacis} function now refits the non-fitting curves with the 'bilinear' method (see \code{fitaci}), which will always return parameter estimates (for better or worse).

\strong{Summarizing and plotting - } Like \code{fitaci}, the batch utility \code{fitacis} also has a standard plotting method. By default, it will make a single plot for every curve that you fit (thus generating many plots). Alternatively, use the setting \code{how="oneplot"} (see Examples below) for a single plot. The fitted \strong{coefficients} are extracted with \code{coef}, which gives a dataframe where each row represents a fitted curve (the grouping label is also included).

\strong{Adding identifying variables - } after fitting multiple curves, the most logical next step is to analyze the coefficient by some categorical variable (species, treatment, location). You can use the \code{id} argument to store variables from the original dataset in the output. It is important that the 'id' variables take only one value per fitted curve, if this is not the case only the first value of the curve will be stored (this will be rarely useful). See examples.
}
\examples{

# Fit many curves (using an example dataset)
# The bilinear method is much faster, but compare using 'default'!
fits <- fitacis(manyacidat, "Curve", fitmethod="bilinear")
with(coef(fits), plot(Vcmax, Jmax))

# The resulting object is a list, with each component an object as returned by fitaci
# So, we can extract one curve:
fits[[1]]
plot(fits[[1]])

# Plot all curves in separate figures with plot(fits)
# Or, in one plot:
plot(fits, how="oneplot")

# Note that parameters can be passed to plot.acifit. For example,
plot(fits, how="oneplot", what="data", col="blue")
plot(fits, how="oneplot", add=TRUE, what="model", lwd=c(1,1))

# Other elements can be summarized with sapply. For example, look at the RMSE:
rmses <- sapply(fits, "[[", "RMSE")
plot(rmses, type='h', ylab="RMSE", xlab="Curve nr")

# And plot the worst-fitting curve:
plot(fits[[which.max(rmses)]])

# It is very straightforward to summarize the coefficients by a factor variable
# that was contained in the original data. In manyacidat, there is a factor variable
# 'treatment'.
# We first have to refit the curves, using the 'id' argument:
fits <- fitacis(manyacidat, "Curve", fitmethod="bilinear", id="treatment")

# And now use this to plot Vcmax by treatment.
boxplot(Vcmax ~ treatment, data=coef(fits), ylim=c(0,130))

}
\references{
Duursma, R.A., 2015. Plantecophys - An R Package for Analysing and Modelling Leaf Gas Exchange Data. PLoS ONE 10, e0143346. doi:10.1371/journal.pone.0143346
}

