% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getBasalAreas.R
\name{getBasalAreas}
\alias{getBasalAreas}
\title{Calculates basal area for each species in a quadrat}
\usage{
getBasalAreas(
  dat,
  inv,
  species = "Species",
  quad = "Quad",
  site = "Site",
  year = "Year",
  geometry = "geometry",
  ...
)
}
\arguments{
\item{dat}{An sf data.frame in which each row represents a unique polygon
(either a genet or a ramet) in a unique site/quadrat/year combination. A
data.frame returned by \code{\link{trackSpp}} can be put directly into this
function. However, it is not necessary for 'dat' to have demographic data or
unique identifiers (i.e. 'trackIDs') assigned. 'dat' must have columns that
contain...
\itemize{
\item a unique identification for each research site in character format
with no NAs (the default column name is "Site")
\item species name in character format with no NAs (the default column
name is "Species")
\item unique quadrat identifier in character format with no NAs (the default
column name is "Quad")
\item year of data collection in integer format with no NAs (the
default column name is "Year")
\item an s.f 'geometry' column that contains a polygon or multipolygon data type
for each individual observation (the default column name is "geometry")
}

This function should only be used if
the individuals in 'dat' were mapped as polygons that are representative of
their actual basal area, i.e. were not mapped as points.}

\item{inv}{The name of each element of the list is a
quadrat name in 'dat', and the contents of that list element is a numeric
vector of all of the years in which that quadrat (or other unique spatial
area) was sampled. Make sure this is the years the quadrat was actually
sampled, not just the years that have data in the 'dat' argument! This
argument allows the function to differentiate between years when the quadrat
wasn't sampled and years when there just weren't any individuals of a species
present in that quadrat.}

\item{species}{An optional character string argument. Indicates
the name of the column in 'dat' that contains species name data. It is
unnecessary to include a value for this argument if the column name is
"Species" (default value is 'Species').}

\item{quad}{An optional character string argument. Indicates
the name of the column in 'dat' that contains quadrat name data. It is
unnecessary to include a value for this argument if the column name is
"Quad" (default is 'Quad').}

\item{site}{An optional character string argument. Indicates
the name of the column in 'dat' that contains site name data. It is
unnecessary to include a value for this argument if the column name is
"Site" (default value is 'Site').}

\item{year}{An optional character string argument. Indicates
the name of the column in 'dat' that contains data for year of sampling. It
is unnecessary to include a value for this argument if the column name is
"Year" (default is 'Year').}

\item{geometry}{An optional character string argument. Indicates
the name of the column in 'dat' that contains sf geometry data. It is
unnecessary to include a value for this argument if the column name is
"geometry" (default is 'geometry').}

\item{...}{Other arguments passed on to methods. Not currently used.}
}
\value{
\code{\link[=getBasalAreas]{getBasalAreas()}} returns a data.frame with the columns "Site",
"Quad", "Year" and "Species". No two rows will have the same values for all
four of these columns. The column "absolute_basalArea" has the area of the
quadrat that is occupied by a species in a given unique site/quadrat/year
combination. This is in the same units as the area in 'dat'.
"quad_basalArea" gives the combined basal area of all organisms in a given
site/quadrat/year. "percent_basalArea" gives the percentage of occupied basal
area within a quadrat that is occupied by each species in a given
site/quadrat/year. For example, species A has a basal area of 22 cm^2 in
quadrat 1 in 2005 ("absolute_basalArea = 22). In 2005, there are 50 cm^2 of
quadrat 1 that are occupied by organisms ("quad_basalArea" = 55). 44\% of the
occupied basal area in quadrat 1 in 2005 is occupied by species A
("percent_basalArea" = 44). There may be an 'NA' in the "percent_basalArea"
column if the "quad_basalArea" for that species and year is 0.
}
\description{
This function calculates the total basal area for each species
present in a quadrat for each year of data collection. Both the absolute
basal area (in the same units of area as the input data.frame) is returned,
as well as the percentage of the total occupied basal area in the quadrat
that is occupied by a given species ("percent total basal area"). If you'd
like to ultimately calculate the population growth rate (lambda) for each
species, you can use the \code{\link{getLambda}} function directly, which
takes raw species occurrence data (like in 'dat' here) and returns lambda
values for each species and quadrat for each transition in the dataset.
This function should only be used if the individuals in 'dat' were mapped as
polygons that are representative of their actual basal area, i.e. were not
mapped as points.
}
\examples{
dat <- grasslandData[grasslandData$Site == "AZ" &
 grasslandData$Year \%in\% c(1922:1925),]
names(dat)[1] <- "speciesName"
inv <- grasslandInventory[unique(dat$Quad)]
outDat <- trackSpp(dat = dat,
 inv = inv,
 dorm = 1,
 buff = .05,
 buffGenet = 0.005,
 clonal = data.frame("Species" = unique(dat$speciesName),
 "clonal" = c(TRUE,FALSE)),
 species = "speciesName",
 aggByGenet = TRUE
 )
getBasalAreas(dat = outDat, inv = inv,
species = "speciesName")
}
\seealso{
\code{\link[=getLambda]{getLambda()}}, which uses this function to calculate basal areas and
ultimately return population growth rates (lambdas) for each species in
each quadrat.
}
