% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PCDFunctions.r
\name{NASbastri}
\alias{NASbastri}
\title{The vertices of the Arc Slice (AS) Proximity Region in the basic triangle}
\usage{
NASbastri(pt, c1, c2, M = "CC", rv = NULL, dec = 4)
}
\arguments{
\item{pt}{A 2D point whose AS proximity region is to be computed}

\item{c1, c2}{Positive real numbers representing the top vertex in basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))},
\code{c1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}}

\item{M}{"CC" which stands for circumcenter of the triangle \eqn{T_b} or a 2D point in Cartesian coordinates or
a 3D point in barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
default is \code{M}="CC" i.e. the circumcenter of \eqn{T_b}}

\item{rv}{The index of the \code{M}-vertex region containing the point, either 1, 2, 3 or \code{NULL} (default is \code{NULL})}

\item{dec}{a positive integer the number of decimals (default is 4) to round the barycentric coordinates when checking whether
the end points fall on the boundary of the triangle \eqn{T_b} or not}
}
\value{
A list with the elements
\item{L,R}{The end points of the line segments on the boundary of the AS proximity region.
 Each row in \code{L} and \code{R} constitute a line segment on the boundary. }
\item{Arc.Slices}{The end points of the arc-slices on the circular parts of the AS proximity region.
Here points in row 1 and row 2 constitute the end points of one arc-slice, points on row 3 and row 4
constitute the end points for the next arc-slice and so on}
}
\description{
Returns the end points of the line segments and arc-slices that constitute the boundary of
AS proximity region for a point in the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}
where \code{c1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.

Vertex regions are based on the center \code{M}="CC" for circumcenter of \eqn{T_b}; or \eqn{M=(m_1,m_2)} in Cartesian
coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of \eqn{T_b};
default is \code{M}="CC" the circumcenter of \eqn{T_b}.
\code{rv} is the index of the vertex region \code{pt} resides, with default=\code{NULL}.

If \code{pt} is outside \eqn{T_b}, it returns \code{NULL} for the proximity region.
\code{dec} is the number of decimals (default is 4) to round the barycentric coordinates when checking whether
the end points fall on the boundary of the triangle \eqn{T_b} or not (so as not to miss the intersection points
due to precision in the decimals).

Any given triangle can be mapped to the basic triangle
by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
preserving uniformity of the points in the original triangle. Hence basic triangle is useful for simulation
studies under the uniformness hypothesis.

See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
}
\examples{
c1<-.4; c2<-.6  #try also c1<-.2; c2<-.2;
A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
Tb<-rbind(A,B,C)

M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)

P1<-as.numeric(runif.bastri(1,c1,c2)$g);  #try also P1<-c(.3,.2)
NASbastri(P1,c1,c2)  #default with M="CC"
NASbastri(P1,c1,c2,M)

#or try
Rv<-rv.bastriCC(P1,c1,c2)$rv
NASbastri(P1,c1,c2,M,Rv)

NASbastri(c(3,5),c1,c2,M)

P2<-c(.5,.4)
NASbastri(P2,c1,c2,M)

P3<-c(1.5,.4)
NASbastri(P3,c1,c2,M)

if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#need to run this when M is given in barycentric coordinates

#plot of the NAS region
P1<-as.numeric(runif.bastri(1,c1,c2)$g);
CC<-circ.cent.bastri(c1,c2)

if (isTRUE(all.equal(M,CC)) || identical(M,"CC"))
{cent<-CC
D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
Ds<-rbind(D1,D2,D3)
cent.name<-"CC"
rv<-rv.bastriCC(P1,c1,c2)$rv
} else
{cent<-M
cent.name<-"M"
Ds<-cp2e.bastri(c1,c2,M)
D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
rv<-rv.bastri.cent(P1,c1,c2,M)$rv
}
RV<-Tb[rv,]
rad<-Dist(P1,RV)

Int.Pts<-NASbastri(P1,c1,c2,M)

Xlim<-range(Tb[,1],P1[1]+rad,P1[1]-rad)
Ylim<-range(Tb[,2],P1[2]+rad,P1[2]-rad)
xd<-Xlim[2]-Xlim[1]
yd<-Ylim[2]-Ylim[1]

plot(A,pch=".",asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
polygon(Tb)
points(rbind(Tb,P1,rbind(Int.Pts$L,Int.Pts$R)))
L<-rbind(cent,cent,cent); R<-Ds
segments(L[,1], L[,2], R[,1], R[,2], lty=2)
interp::circles(P1[1],P1[2],rad,lty=2)
L<-Int.Pts$L; R<-Int.Pts$R
segments(L[,1], L[,2], R[,1], R[,2], lty=1,col=2)
Arcs<-Int.Pts$A;
if (!is.null(Arcs))
{
  K<-nrow(Arcs)/2
  for (i in 1:K)
  {A1<-Arcs[2*i-1,]; A2<-Arcs[2*i,];
  angles<-angle.str2end(A1,P1,A2)$c

  plotrix::draw.arc(P1[1],P1[2],rad,angle1=angles[1],angle2=angles[2],col=2)
  }
}

#proximity region with the triangle (i.e., for labeling the vertices of the NAS)
IP.txt<-intpts<-c()
if (!is.null(Int.Pts$A))
{
  intpts<-unique(round(Int.Pts$A,7))
  #this part is for labeling the intersection points of the spherical
  for (i in 1:(length(intpts)/2))
    IP.txt<-c(IP.txt,paste("I",i+1, sep = ""))
}
txt<-rbind(Tb,P1,cent,intpts)
txt.str<-c("A","B","C","P1",cent.name,IP.txt)
text(txt+cbind(rep(xd*.02,nrow(txt)),rep(-xd*.03,nrow(txt))),txt.str)

c1<-.4; c2<-.6;
P1<-c(.3,.2)
NASbastri(P1,c1,c2,M)

}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{NAStri}} and \code{\link{IndNASbastri}}
}
