#' @include http_status.R

# Returns an Error object.
Error <- struct(
  code = "",
  message = "",
  status_code = NA_integer_,
  error_response = list()
)

ERROR_MSG_TEMPLATE <- "An error occurred when calling the %s operation%s: %s"

serialization_error <- function(request) {
  error_message <- http_statuses[[as.character(request$http_response$status_code)]]
  Error(
    "SerializationError",
    sprintf(
      ERROR_MSG_TEMPLATE,
      request$operation$name,
      get_retry_info(request$retry_count),
      error_message
    ),
    request$http_response$status_code
  )
}

get_retry_info <- function(retry_count) {
  if (retry_count) {
    retry_msg <- sprintf(" (reached max retries: %d)", retry_count)
  } else {
    retry_msg <- ""
  }
  return(retry_msg)
}

#' Generate a classed http error
#'
#' This function generates S3 error objects which are passed to
#' [stop()] to generate classed errors.
#' These can be used in conjunction with [tryCatch()]
#' to respond differently to different types of failure.
#'
#' This expands on the condition messages generated by the httr package
#' in `httr::http_condition()`. In addition to error classes like
#' `http_error`, `http_400`, and `http_404`, this sets attributes
#' on the error condition that contain the unmarshalled error
#' response from AWS for the respective service.
#'
#' @keywords internal
#' @return An S3 object that inherits from (e.g.) condition, error,
#'   http_error, http_400 and http_404.
#' @param e An error struct, like ones generated by `paws::Error`.
#'   Must contain fields `code` (string),`message` (string),
#'   `status_code` (numeric), `error_response` (list).
#' @param call The call stored in the condition object.
#' @seealso
#'   <http://adv-r.had.co.nz/Exceptions-Debugging.html#condition-handling>
#'   for more details about R's condition handling model
#'   https://github.com/r-lib/httr/blob/master/R/response-status.r#L310
#'   for the httr inspiration of this condition
#'
aws_error <- function(e, call = sys.call(-1), use_call = FALSE) {
  full_message <- sprintf("%s (HTTP %d). %s", e$code, e$status_code, e$message)

  status_type <- (e$status_code %/% 100) * 100
  if (is.na(e$status_code)) {
    # If the http_error is still NA (default), don't set specific classes.
    http_class <- c("http_error")
  } else {
    http_class <- paste0("http_", unique(c(e$status_code, status_type, "error")))
  }
  if (!use_call) {
    call <- NULL
  }

  structure(
    list(
      message = full_message,
      call = call,
      status_code = e$status_code,
      error_response = e$error_response
    ),
    class = c("paws_error", http_class, "error", "condition"),
    status_code = e$status_code,
    error_response = e$error_response
  )
}
