% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adp.R
\docType{methods}
\name{setFlags,adp-method}
\alias{setFlags,adp-method}
\title{Set data-quality flags within a adp object}
\usage{
\S4method{setFlags}{adp}(object, name = NULL, i = NULL, value = NULL,
  debug = getOption("oceDebug"))
}
\arguments{
\item{object}{An oce object.}

\item{name}{Character string indicating the name of the variable to be flagged. If
this variable is not contained in the object's \code{data} slot, an error is reported.}

\item{i}{Indication of where to insert the flags; see \dQuote{Description} for
general rules and \dQuote{Details} for rules for \code{\link{adp-class}}
objects.}

\item{value}{The value to be inserted in the flag.}

\item{debug}{Integer set to 0 for quiet action or to 1 for some debugging.}
}
\value{
An object with flags set as indicated.
}
\description{
This function changes specified entries in the data-quality
flags of a adp object, which are stored within
a list named \code{flags} that resides in the \code{metadata}
slot. If the object already has a flag set up for \code{name},
then only the specified entries are altered. If not, the flag
entry is first created and its entries set to \code{default},
after which the entries specified by \code{i}
are changed to \code{value}.

The specification is made with \code{i}, the form of which
is determined by the data item in question. Generally,
the rules are as follows:
\enumerate{
\item If the data item is a vector, then \code{i} must be (a)
an integer vector specifying indices to be set to \code{value},
(b) a logical vector of length matching the data item, with
\code{TRUE} meaning to set the flag to \code{value}, or (c)
a function that takes an \code{oce} object as its single
argument, and returns a vector in either of the forms
just described.
\item If the data item is an array, then \code{i} must be
(a) a data frame of integers whose rows specify spots to change
(where the number of columns matches the number of dimensions
of the data item), (b) a logical array that has dimension equal to
that of the data item, or (c) a function that takes an \code{oce}
object as its single input and returns such a data frame or array.
}
See \dQuote{Details} for the particular case of
\code{\link{adp-class}} objects.
}
\details{
The only flag that may be set is \code{v}, for the array holding velocity. See \dQuote{Indexing rules}, noting that adp data are stored in 3D arrays; Example 1 shows using a data frame for \code{i}, while Example 2 shows using an array.
}
\section{Caution}{

This function was added in early May, 2018, and is likely to undergo
changes until the mid-summer of that year.  Use with caution.
}

\examples{
library(oce)
data(adp)

## Example 1: flag first 10 samples in a mid-depth bin of beam 1
i1 <- data.frame(1:20, 40, 1)
adpQC <- initializeFlags(adp, "v", 2)
adpQC <- setFlags(adpQC, "v", i1, 3)
adpClean1 <- handleFlags(adpQC, flags=list(3), actions=list("NA"))
par(mfrow=c(2, 1))
## Top: original, bottom: altered
plot(adp, which="u1")
plot(adpClean1, which="u1")

## Example 2: percent-good and error-beam scheme
v <- adp[["v"]]
i2 <- array(FALSE, dim=dim(v))
g <- adp[["g", "numeric"]]
# Thresholds on percent "goodness" and error "velocity"
G <- 25
V4 <- 0.45
for (k in 1:3)
    i2[,,k] <- ((g[,,k]+g[,,4]) < G) | (v[,,4] > V4)
adpQC2 <- initializeFlags(adp, "v", 2)
adpQC2 <- setFlags(adpQC2, "v", i2, 3)
adpClean2 <- handleFlags(adpQC2, flags=list(3), actions=list("NA"))
## Top: original, bottom: altered
plot(adp, which="u1")
plot(adpClean2, which="u1") # differs at 8h and 20h

}
\seealso{
Other functions relating to data-quality flags: \code{\link{defaultFlags}},
  \code{\link{handleFlags,adp-method}},
  \code{\link{handleFlags,argo-method}},
  \code{\link{handleFlags,ctd-method}},
  \code{\link{handleFlags,section-method}},
  \code{\link{handleFlags}},
  \code{\link{initializeFlagScheme,ctd-method}},
  \code{\link{initializeFlagScheme,oce-method}},
  \code{\link{initializeFlagScheme,section-method}},
  \code{\link{initializeFlagSchemeInternal}},
  \code{\link{initializeFlagScheme}},
  \code{\link{initializeFlags,adp-method}},
  \code{\link{initializeFlags,oce-method}},
  \code{\link{initializeFlagsInternal}},
  \code{\link{initializeFlags}},
  \code{\link{setFlags,ctd-method}},
  \code{\link{setFlags,oce-method}}, \code{\link{setFlags}}

Other things related to adp data: \code{\link{[[,adp-method}},
  \code{\link{[[<-,adp-method}},
  \code{\link{ad2cpHeaderValue}}, \code{\link{adp-class}},
  \code{\link{adpEnsembleAverage}}, \code{\link{adp}},
  \code{\link{as.adp}}, \code{\link{beamName}},
  \code{\link{beamToXyzAdpAD2CP}},
  \code{\link{beamToXyzAdp}}, \code{\link{beamToXyzAdv}},
  \code{\link{beamToXyz}}, \code{\link{beamUnspreadAdp}},
  \code{\link{binmapAdp}}, \code{\link{enuToOtherAdp}},
  \code{\link{enuToOther}},
  \code{\link{handleFlags,adp-method}},
  \code{\link{is.ad2cp}}, \code{\link{plot,adp-method}},
  \code{\link{read.adp.ad2cp}},
  \code{\link{read.adp.nortek}},
  \code{\link{read.adp.rdi}},
  \code{\link{read.adp.sontek.serial}},
  \code{\link{read.adp.sontek}}, \code{\link{read.adp}},
  \code{\link{read.aquadoppHR}},
  \code{\link{read.aquadoppProfiler}},
  \code{\link{read.aquadopp}}, \code{\link{rotateAboutZ}},
  \code{\link{subset,adp-method}},
  \code{\link{summary,adp-method}}, \code{\link{toEnuAdp}},
  \code{\link{toEnu}}, \code{\link{velocityStatistics}},
  \code{\link{xyzToEnuAdpAD2CP}},
  \code{\link{xyzToEnuAdp}}, \code{\link{xyzToEnu}}
}
\concept{functions relating to data-quality flags}
\concept{things related to adp data}
