% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_nfl.R
\name{simulate_nfl}
\alias{simulate_nfl}
\title{Simulate an NFL Season}
\usage{
simulate_nfl(
  nfl_season = NULL,
  process_games = NULL,
  ...,
  playoff_seeds = ifelse(nfl_season >= 2020, 7, 6),
  if_ended_today = FALSE,
  fresh_season = FALSE,
  fresh_playoffs = FALSE,
  tiebreaker_depth = 3,
  test_week = NULL,
  simulations = 1000,
  sims_per_round = max(ceiling(simulations/future::availableCores() * 2), 100),
  .debug = FALSE,
  print_summary = FALSE
)
}
\arguments{
\item{nfl_season}{Season to simulate}

\item{process_games}{A function to estimate and simulate the results of games. Uses team,
schedule, and week number as arguments.}

\item{...}{Additional parameters passed on to the function \code{process_games}.}

\item{playoff_seeds}{Number of playoff teams per conference (increased
in 2020 from 6 to 7).}

\item{if_ended_today}{Either \code{TRUE} or \code{FALSE}. If TRUE, ignore remaining regular
season games and cut to playoffs based on current regular season data.}

\item{fresh_season}{Either \code{TRUE} or \code{FALSE}. Whether to blank out all game results
and simulate the the season from scratch (TRUE) or take game results so far as a given
and only simulate the rest (FALSE).}

\item{fresh_playoffs}{Either \code{TRUE} or \code{FALSE}. Whether to blank out all playoff
game results and simulate the postseason from scratch (TRUE) or take game results so far
as a given and only simulate the rest (FALSE).}

\item{tiebreaker_depth}{A single value equal to 1, 2, or 3. The default is 3. The
value controls the depth of tiebreakers that shall be applied. The deepest
currently implemented tiebreaker is strength of schedule. The following
values are valid:
\describe{
\item{tiebreaker_depth = 1}{Break all ties with a coinflip. Fastest variant.}
\item{tiebreaker_depth = 2}{Apply head-to-head and division win percentage tiebreakers. Random if still tied.}
\item{tiebreaker_depth = 3}{Apply all tiebreakers through strength of schedule. Random if still tied.}
}}

\item{test_week}{Aborts after the simulator reaches this week and returns the results
from your process games call.}

\item{simulations}{Equals the number of times the given NFL season shall be simulated}

\item{sims_per_round}{The number of \code{simulations} can be split into
multiple rounds and be processed parallel. This parameter controls the number
of simulations per round. The default value determines the number of
locally available cores and calculates the number of simulations per round
to be equal to half of the available cores (various benchmarks showed this
results in optimal performance).}

\item{.debug}{Either \code{TRUE} or \code{FALSE}. Controls whether additional
messages are printed to the console showing what the tie-breaking algorithms
are currently performing.}

\item{print_summary}{If \code{TRUE}, prints the summary statistics to the console.}
}
\value{
An \code{nflseedR_simulation} object containing a list of 6 data frames
data frames with the results of all simulated games,
the final standings in each simulated season (incl. playoffs and draft order),
summary statistics across all simulated seasons, and the siumulation parameters. For a full list,
please see \href{https://nflseedr.com/articles/articles/nflsim.html#simulation-output}{the package website}.
}
\description{
This function simulates a given NFL season multiple times using custom functions
to estimate and simulate game results and computes the outcome of the given
season including playoffs and draft order.
It is possible to run the function in parallel processes by calling the
appropriate \link[future]{plan}.
Progress updates can be activated by calling \link[progressr]{handlers}
before the start of the simulations.
Please see the below given section "Details" for further information.
}
\details{
\subsection{More Speed Using Parallel Processing}{

We recommend choosing a default parallel processing method and saving it
as an environment variable in the R user profile to make sure all futures
will be resolved with the chosen method by default.
This can be done by following the below given steps.

First, run the following line and the user profile should be opened automatically.
If you haven't saved any environment variables yet, this will be an empty file.

\if{html}{\out{<div class="sourceCode">}}\preformatted{usethis::edit_r_environ()
}\if{html}{\out{</div>}}

In the opened file add the next line, then save the file and restart your R session.
Please note that this example sets "multisession" as default. For most users
this should be the appropriate plan but please make sure it truly is.

\if{html}{\out{<div class="sourceCode">}}\preformatted{R_FUTURE_PLAN="multisession"
}\if{html}{\out{</div>}}

After the session is freshly restarted please check if the above method worked
by running the next line. If the output is \code{FALSE} you successfully set up a
default non-sequential \code{\link[future:plan]{future::plan()}}. If the output is \code{TRUE} all functions
will behave like they were called with \code{\link[purrr:map]{purrr::map()}} and NOT in multisession.

\if{html}{\out{<div class="sourceCode">}}\preformatted{inherits(future::plan(), "sequential")
}\if{html}{\out{</div>}}

For more information on possible plans please see
\href{https://github.com/HenrikBengtsson/future/blob/develop/README.md}{the future package Readme}.
}

\subsection{Get Progress Updates while Functions are Running}{

Most nflfastR functions are able to show progress updates
using \code{\link[progressr:progressor]{progressr::progressor()}} if they are turned on before the function is
called. There are at least two basic ways to do this by either activating
progress updates globally (for the current session) with

\if{html}{\out{<div class="sourceCode">}}\preformatted{progressr::handlers(global = TRUE)
}\if{html}{\out{</div>}}

or by piping the function call into \code{\link[progressr:with_progress]{progressr::with_progress()}}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{simulate_nfl(2020, fresh_season = TRUE) \%>\%
  progressr::with_progress()
}\if{html}{\out{</div>}}

For more information how to work with progress handlers please see \link[progressr:progressr]{progressr::progressr}.
}
}
\examples{
\donttest{
library(nflseedR)

# Activate progress updates
# progressr::handlers(global = TRUE)

# Parallel processing can be activated via the following line
# future::plan("multisession")

# Simulate the season 4 times in 2 rounds
sim <- nflseedR::simulate_nfl(
  nfl_season = 2020,
  fresh_season = TRUE,
  simulations = 4,
  sims_per_round = 2
)

# Overview output
dplyr::glimpse(sim)
}
}
\seealso{
The examples \href{https://nflseedr.com/articles/articles/nflsim.html}{on the package website}

The method \code{\link[=summary.nflseedR_simulation]{summary.nflseedR_simulation()}} that creates a pretty html summary table.
}
