% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parsnip-arima_reg.R
\name{arima_reg}
\alias{arima_reg}
\title{General Interface for ARIMA Regression Models}
\usage{
arima_reg(
  mode = "regression",
  seasonal_period = NULL,
  non_seasonal_ar = NULL,
  non_seasonal_differences = NULL,
  non_seasonal_ma = NULL,
  seasonal_ar = NULL,
  seasonal_differences = NULL,
  seasonal_ma = NULL
)
}
\arguments{
\item{mode}{A single character string for the type of model.
The only possible value for this model is "regression".}

\item{seasonal_period}{A seasonal frequency. Uses "auto" by default.
A character phrase of "auto" or time-based phrase of "2 weeks"
can be used if a date or date-time variable is provided.
See Fit Details below.}

\item{non_seasonal_ar}{The order of the non-seasonal auto-regressive (AR) terms. Often denoted "p" in pdq-notation.}

\item{non_seasonal_differences}{The order of integration for non-seasonal differencing. Often denoted "d" in pdq-notation.}

\item{non_seasonal_ma}{The order of the non-seasonal moving average (MA) terms. Often denoted "q" in pdq-notation.}

\item{seasonal_ar}{The order of the seasonal auto-regressive (SAR) terms. Often denoted "P" in PDQ-notation.}

\item{seasonal_differences}{The order of integration for seasonal differencing. Often denoted "D" in PDQ-notation.}

\item{seasonal_ma}{The order of the seasonal moving average (SMA) terms. Often denoted "Q" in PDQ-notation.}
}
\description{
\code{arima_reg()} is a way to generate a \emph{specification} of an ARIMA model
before fitting and allows the model to be created using
different packages. Currently the only package is \code{forecast}.
}
\details{
The data given to the function are not saved and are only used
to determine the \emph{mode} of the model. For \code{arima_reg()}, the
mode will always be "regression".

The model can be created using the \code{fit()} function using the
following \emph{engines}:
\itemize{
\item "auto_arima" (default) - Connects to \code{\link[forecast:auto.arima]{forecast::auto.arima()}}
\item "arima" - Connects to \code{\link[forecast:Arima]{forecast::Arima()}}
}

\strong{Main Arguments}

The main arguments (tuning parameters) for the model are:
\itemize{
\item \code{seasonal_period}: The periodic nature of the seasonality. Uses "auto" by default.
\item \code{non_seasonal_ar}: The order of the non-seasonal auto-regressive (AR) terms.
\item \code{non_seasonal_differences}: The order of integration for non-seasonal differencing.
\item \code{non_seasonal_ma}: The order of the non-seasonal moving average (MA) terms.
\item \code{seasonal_ar}: The order of the seasonal auto-regressive (SAR) terms.
\item \code{seasonal_differences}: The order of integration for seasonal differencing.
\item \code{seasonal_ma}: The order of the seasonal moving average (SMA) terms.
}

These arguments are converted to their specific names at the
time that the model is fit.

Other options and argument can be
set using \code{set_engine()} (See Engine Details below).

If parameters need to be modified, \code{update()} can be used
in lieu of recreating the object from scratch.
}
\section{Engine Details}{


The standardized parameter names in \code{modeltime} can be mapped to their original
names in each engine:\tabular{lll}{
   modeltime \tab forecast::auto.arima \tab forecast::Arima \cr
   seasonal_period \tab ts(frequency) \tab ts(frequency) \cr
   non_seasonal_ar, non_seasonal_differences, non_seasonal_ma \tab max.p, max.d, max.q \tab order = c(p,d,q) \cr
   seasonal_ar, seasonal_differences, seasonal_ma \tab max.P, max.D, max.Q \tab seasonal = c(P,D,Q) \cr
}


Other options can be set using \code{set_engine()}.

\strong{auto_arima (default engine)}

The engine uses \code{\link[forecast:auto.arima]{forecast::auto.arima()}}.

Function Parameters:\preformatted{## function (y, d = NA, D = NA, max.p = 5, max.q = 5, max.P = 2, max.Q = 2, 
##     max.order = 5, max.d = 2, max.D = 1, start.p = 2, start.q = 2, start.P = 1, 
##     start.Q = 1, stationary = FALSE, seasonal = TRUE, ic = c("aicc", "aic", 
##         "bic"), stepwise = TRUE, nmodels = 94, trace = FALSE, approximation = (length(x) > 
##         150 | frequency(x) > 12), method = NULL, truncate = NULL, xreg = NULL, 
##     test = c("kpss", "adf", "pp"), test.args = list(), seasonal.test = c("seas", 
##         "ocsb", "hegy", "ch"), seasonal.test.args = list(), allowdrift = TRUE, 
##     allowmean = TRUE, lambda = NULL, biasadj = FALSE, parallel = FALSE, 
##     num.cores = 2, x = y, ...)
}

The \emph{MAXIMUM} nonseasonal ARIMA terms (\code{max.p}, \code{max.d}, \code{max.q}) and
seasonal ARIMA terms (\code{max.P}, \code{max.D}, \code{max.Q}) are provided to
\code{\link[forecast:auto.arima]{forecast::auto.arima()}} via \code{arima_reg()} parameters.
Other options and argument can be set using \code{set_engine()}.

Parameter Notes:
\itemize{
\item All values of nonseasonal pdq and seasonal PDQ are maximums.
The \code{forecast::auto.arima()} model will select a value using these as an upper limit.
\item \code{xreg} - This is supplied via the parsnip / modeltime \code{fit()} interface
(so don't provide this manually). See Fit Details (below).
}

\strong{arima}

The engine uses \code{\link[forecast:Arima]{forecast::Arima()}}.

Function Parameters:\preformatted{## function (y, order = c(0, 0, 0), seasonal = c(0, 0, 0), xreg = NULL, include.mean = TRUE, 
##     include.drift = FALSE, include.constant, lambda = model$lambda, biasadj = FALSE, 
##     method = c("CSS-ML", "ML", "CSS"), model = NULL, x = y, ...)
}

The nonseasonal ARIMA terms (\code{order}) and seasonal ARIMA terms (\code{seasonal})
are provided to \code{\link[forecast:Arima]{forecast::Arima()}} via \code{arima_reg()} parameters.
Other options and argument can be set using \code{set_engine()}.

Parameter Notes:
\itemize{
\item \code{xreg} - This is supplied via the parsnip / modeltime \code{fit()} interface
(so don't provide this manually). See Fit Details (below).
\item \code{method} - The default is set to "ML" (Maximum Likelihood).
This method is more robust at the expense of speed and possible
selections may fail unit root inversion testing. Alternatively, you can add \code{method = "CSS-ML"} to
evaluate Conditional Sum of Squares for starting values, then Maximium Likelihood.
}
}

\section{Fit Details}{


\strong{Date and Date-Time Variable}

It's a requirement to have a date or date-time variable as a predictor.
The \code{fit()} interface accepts date and date-time features and handles them internally.
\itemize{
\item \code{fit(y ~ date)}
}

\emph{Seasonal Period Specification}

The period can be non-seasonal (\verb{seasonal_period = 1 or "none"}) or
yearly seasonal (e.g. For monthly time stamps, \code{seasonal_period = 12}, \code{seasonal_period = "12 months"}, or \code{seasonal_period = "yearly"}).
There are 3 ways to specify:
\enumerate{
\item \code{seasonal_period = "auto"}: A seasonal period is selected based on the periodicity of the data (e.g. 12 if monthly)
\item \code{seasonal_period = 12}: A numeric frequency. For example, 12 is common for monthly data
\item \code{seasonal_period = "1 year"}: A time-based phrase. For example, "1 year" would convert to 12 for monthly data.
}

\strong{Univariate (No xregs, Exogenous Regressors):}

For univariate analysis, you must include a date or date-time feature. Simply use:
\itemize{
\item Formula Interface (recommended): \code{fit(y ~ date)} will ignore xreg's.
\item XY Interface: \code{fit_xy(x = data[,"date"], y = data$y)} will ignore xreg's.
}

\strong{Multivariate (xregs, Exogenous Regressors)}

The \code{xreg} parameter is populated using the \code{fit()} or \code{fit_xy()} function:
\itemize{
\item Only \code{factor}, \verb{ordered factor}, and \code{numeric} data will be used as xregs.
\item Date and Date-time variables are not used as xregs
\item \code{character} data should be converted to factor.
}

\emph{Xreg Example:} Suppose you have 3 features:
\enumerate{
\item \code{y} (target)
\item \code{date} (time stamp),
\item \code{month.lbl} (labeled month as a ordered factor).
}

The \code{month.lbl} is an exogenous regressor that can be passed to the \code{arima_reg()} using
\code{fit()}:
\itemize{
\item \code{fit(y ~ date + month.lbl)} will pass \code{month.lbl} on as an exogenous regressor.
\item \code{fit_xy(data[,c("date", "month.lbl")], y = data$y)} will pass x, where x is a data frame containing \code{month.lbl}
and the \code{date} feature. Only \code{month.lbl} will be used as an exogenous regressor.
}

Note that date or date-time class values are excluded from \code{xreg}.
}

\examples{
library(dplyr)
library(parsnip)
library(rsample)
library(timetk)
library(modeltime)

# Data
m750 <- m4_monthly \%>\% filter(id == "M750")
m750

# Split Data 80/20
splits <- initial_time_split(m750, prop = 0.8)

# ---- AUTO ARIMA ----

# Model Spec
model_spec <- arima_reg() \%>\%
    set_engine("auto_arima")

# Fit Spec
model_fit <- model_spec \%>\%
    fit(log(value) ~ date, data = training(splits))
model_fit


# ---- STANDARD ARIMA ----

# Model Spec
model_spec <- arima_reg(
        seasonal_period          = 12,
        non_seasonal_ar          = 3,
        non_seasonal_differences = 1,
        non_seasonal_ma          = 3,
        seasonal_ar              = 1,
        seasonal_differences     = 0,
        seasonal_ma              = 1
    ) \%>\%
    set_engine("arima")

# Fit Spec
model_fit <- model_spec \%>\%
    fit(log(value) ~ date, data = training(splits))
model_fit

}
\seealso{
\code{\link[=fit.model_spec]{fit.model_spec()}}, \code{\link[=set_engine]{set_engine()}}
}
