% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mmcif.R
\name{mmcif_fit}
\alias{mmcif_fit}
\title{Fits a Mixed Competing Risk Model}
\usage{
mmcif_fit(
  par,
  object,
  n_threads = 1L,
  control.outer = list(itmax = 100L, method = "nlminb", kkt2.check = FALSE, trace =
    FALSE),
  control.optim = list(eval.max = 10000L, iter.max = 10000L),
  ghq_data = object$ghq_data,
  ...
)
}
\arguments{
\item{par}{numeric vector with parameters. This is using a log
Cholesky decomposition for the covariance matrix.}

\item{object}{an object from \code{\link{mmcif_data}}.}

\item{n_threads}{the number of threads to use.}

\item{control.outer, control.optim, ...}{arguments passed to
\code{\link[alabama]{auglag}}.}

\item{ghq_data}{the Gauss-Hermite quadrature nodes and weights to use.
It should be a list with two elements called \code{"node"} and \code{"weight"}.
The argument can also be a list with lists with different sets of quadrature
nodes. In this case, fits are successively made using the previous fit as the
starting value. This may
reduce the computation time by starting with fewer quadrature nodes.}
}
\value{
The output from \code{\link{auglag}}.
}
\description{
Fits mixed cumulative incidence functions model by maximizing the log
composite likelihood function.
}
\examples{
\donttest{if(require(mets)){
  # prepare the data
  data(prt)

  # truncate the time
  max_time <- 90
  prt <- within(prt, {
    status[time >= max_time] <- 0
    time <- pmin(time, max_time)
  })

  # select the DZ twins and re-code the status
  prt_use <- subset(prt, zyg == "DZ") |>
    transform(status = ifelse(status == 0, 3L, status))

  # randomly sub-sample
  set.seed(1)
  prt_use <- subset(
    prt_use, id \%in\% sample(unique(id), length(unique(id)) \%/\% 10L))

  n_threads <- 2L
  mmcif_obj <- mmcif_data(
    ~ country - 1, prt_use, status, time, id, max_time,
    2L, strata = country)

  # get the staring values
  start_vals <- mmcif_start_values(mmcif_obj, n_threads = n_threads)

  # estimate the parameters
  ests <- mmcif_fit(start_vals$upper, mmcif_obj, n_threads = n_threads)

  # show the estimated covariance matrix of the random effects
  tail(ests$par, 10L) |> log_chol_inv() |> print()

  # gradient is ~ zero
  mmcif_logLik_grad(
    mmcif_obj, ests$par, is_log_chol = TRUE, n_threads = n_threads) |>
    print()
}}

}
\references{
Cederkvist, L., Holst, K. K., Andersen, K. K., &
Scheike, T. H. (2019).
\emph{Modeling the cumulative incidence function of multivariate competing
risks data allowing for within-cluster dependence of risk and timing}.
Biostatistics, Apr 1, 20(2), 199-217.
}
\seealso{
\code{\link{mmcif_data}}, \code{\link{mmcif_start_values}} and
\code{\link{mmcif_sandwich}}.
}
