% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TunerBatchHyperband.R
\name{mlr_tuners_hyperband}
\alias{mlr_tuners_hyperband}
\alias{TunerBatchHyperband}
\title{Tuner Using the Hyperband Algorithm}
\source{
Li L, Jamieson K, DeSalvo G, Rostamizadeh A, Talwalkar A (2018).
\dQuote{Hyperband: A Novel Bandit-Based Approach to Hyperparameter Optimization.}
\emph{Journal of Machine Learning Research}, \bold{18}(185), 1-52.
\url{https://jmlr.org/papers/v18/16-558.html}.
}
\description{
Optimizer using the Hyperband (HB) algorithm.
HB runs the \link[=OptimizerBatchSuccessiveHalving]{Successive Halving Algorithm} (SHA) with different numbers of stating configurations.
The algorithm is initialized with the same parameters as Successive Halving but without \code{n}.
Each run of Successive Halving is called a bracket and starts with a different budget \code{r_0}.
A smaller starting budget means that more configurations can be tried out.
The most explorative bracket allocated the minimum budget \code{r_min}.
The next bracket increases the starting budget by a factor of \code{eta}.
In each bracket, the starting budget increases further until the last bracket \code{s = 0} essentially performs a random search with the full budget \code{r_max}.
The number of brackets \code{s_max + 1} is calculated with \code{s_max = log(r_min / r_max)(eta)}.
Under the condition that \code{r_0} increases by \code{eta} with each bracket, \code{r_min} sometimes has to be adjusted slightly in order not to use more than \code{r_max} resources in the last bracket.
The number of configurations in the base stages is calculated so that each bracket uses approximately the same amount of budget.
The following table shows a full run of HB with \code{eta = 2}, \code{r_min = 1} and \code{r_max = 8}.\tabular{rlrrlrrlrrlrr}{
   \code{s} \tab  \tab  \tab 3 \tab  \tab  \tab 2 \tab  \tab  \tab 1 \tab  \tab  \tab 0 \cr
   \code{i} \tab  \tab \code{n_i} \tab \code{r_i} \tab  \tab \code{n_i} \tab \code{r_i} \tab  \tab \code{n_i} \tab \code{r_i} \tab  \tab \code{n_i} \tab \code{r_i} \cr
   0 \tab  \tab 8 \tab 1 \tab  \tab 6 \tab 2 \tab  \tab 4 \tab 4 \tab  \tab 8 \tab 4 \cr
   1 \tab  \tab 4 \tab 2 \tab  \tab 3 \tab 4 \tab  \tab 2 \tab 8 \tab  \tab  \tab  \cr
   2 \tab  \tab 2 \tab 4 \tab  \tab 1 \tab 8 \tab  \tab  \tab  \tab  \tab  \tab  \cr
   3 \tab  \tab 1 \tab 8 \tab  \tab  \tab  \tab  \tab  \tab  \tab  \tab  \tab  \cr
}


\code{s} is the bracket number, \code{i} is the stage number, \code{n_i} is the number of configurations and \code{r_i} is the budget allocated to a single configuration.

The budget hyperparameter must be tagged with \code{"budget"} in the search space.
The minimum budget (\code{r_min}) which is allocated in the base stage of the most explorative bracket, is set by the lower bound of the budget parameter.
The upper bound defines the maximum budget (\code{r_max}) which is allocated to the candidates in the last stages.
}
\section{Dictionary}{

This \link[mlr3tuning:Tuner]{mlr3tuning::Tuner} can be instantiated via the \link[mlr3misc:Dictionary]{dictionary}
\link[mlr3tuning:mlr_tuners]{mlr3tuning::mlr_tuners} or with the associated sugar function \code{\link[mlr3tuning:tnr]{mlr3tuning::tnr()}}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{TunerBatchHyperband$new()
mlr_tuners$get("hyperband")
tnr("hyperband")
}\if{html}{\out{</div>}}
}

\section{Subsample Budget}{

If the learner lacks a natural budget parameter,
\link[mlr3pipelines:mlr_pipeops_subsample]{mlr3pipelines::PipeOpSubsample} can be applied to use the subsampling rate
as budget parameter. The resulting \link[mlr3pipelines:mlr_learners_graph]{mlr3pipelines::GraphLearner} is fitted on
small proportions of the \link[mlr3:Task]{mlr3::Task} in the first stage, and on the complete
task in last stage.
}

\section{Custom Sampler}{

Hyperband supports custom \link[paradox:Sampler]{paradox::Sampler} object for initial
configurations in each bracket.
A custom sampler may look like this (the full example is given in the
\emph{examples} section):

\if{html}{\out{<div class="sourceCode">}}\preformatted{# - beta distribution with alpha = 2 and beta = 5
# - categorical distribution with custom probabilities
sampler = SamplerJointIndep$new(list(
  Sampler1DRfun$new(params[[2]], function(n) rbeta(n, 2, 5)),
  Sampler1DCateg$new(params[[3]], prob = c(0.2, 0.3, 0.5))
))
}\if{html}{\out{</div>}}
}

\section{Progress Bars}{

\verb{$optimize()} supports progress bars via the package \CRANpkg{progressr}
combined with a \link[bbotk:Terminator]{bbotk::Terminator}. Simply wrap the function in
\code{progressr::with_progress()} to enable them. We recommend to use package
\CRANpkg{progress} as backend; enable with \code{progressr::handlers("progress")}.
}

\section{Parallelization}{

This hyperband implementation evaluates hyperparameter configurations of equal budget across brackets in one batch.
For example, all configurations in stage 1 of bracket 3 and stage 0 of bracket 2 in one batch.
To select a parallel backend, use the \code{plan()} function of the \CRANpkg{future} package.
}

\section{Logging}{

Hyperband uses a logger (as implemented in \CRANpkg{lgr}) from package
\CRANpkg{bbotk}.
Use \code{lgr::get_logger("bbotk")} to access and control the logger.
}

\section{Resources}{

The \href{https://mlr-org.com/gallery-all-optimization.html}{gallery} features a collection of case studies and demos about optimization.
\itemize{
\item \href{https://mlr-org.com/gallery/series/2023-01-15-hyperband-xgboost/}{Tune} the hyperparameters of XGBoost with Hyperband.
\item Use data \href{https://mlr-org.com/gallery/series/2023-01-16-hyperband-subsampling/}{subsampling} and Hyperband to optimize a support vector machine.
}
}

\section{Parameters}{

\describe{
\item{\code{eta}}{\code{numeric(1)}\cr
With every stage, the budget is increased by a factor of \code{eta} and only the best \code{1 / eta} points are promoted to the next stage.
Non-integer values are supported, but \code{eta} is not allowed to be less or equal to 1.}
\item{\code{sampler}}{\link[paradox:Sampler]{paradox::Sampler}\cr
Object defining how the samples of the parameter space should be drawn in the base stage of each bracket.
The default is uniform sampling.}
\item{\code{repetitions}}{\code{integer(1)}\cr
If \code{1} (default), optimization is stopped once all brackets are evaluated.
Otherwise, optimization is stopped after \code{repetitions} runs of HB.
The \link[bbotk:Terminator]{bbotk::Terminator} might stop the optimization before all repetitions are executed.}
}
}

\section{Archive}{

The \link[bbotk:Archive]{bbotk::Archive} holds the following additional columns that are specific to HB:
\itemize{
\item \code{bracket} (\code{integer(1)})\cr
The bracket index. Counts down to 0.
\item \code{stage} (\verb{integer(1))}\cr
The stages of each bracket. Starts counting at 0.
\item \code{repetition} (\verb{integer(1))}\cr
Repetition index. Start counting at 1.
}
}

\examples{
if(requireNamespace("xgboost")) {
  library(mlr3learners)

  # define hyperparameter and budget parameter
  search_space = ps(
    nrounds = p_int(lower = 1, upper = 16, tags = "budget"),
    eta = p_dbl(lower = 0, upper = 1),
    booster = p_fct(levels = c("gbtree", "gblinear", "dart"))
  )

  \donttest{
  # hyperparameter tuning on the pima indians diabetes data set
  instance = tune(
    tnr("hyperband"),
    task = tsk("pima"),
    learner = lrn("classif.xgboost", eval_metric = "logloss"),
    resampling = rsmp("cv", folds = 3),
    measures = msr("classif.ce"),
    search_space = search_space,
    term_evals = 100
  )

  # best performing hyperparameter configuration
  instance$result
  }
}
}
\section{Super classes}{
\code{\link[mlr3tuning:Tuner]{mlr3tuning::Tuner}} -> \code{\link[mlr3tuning:TunerBatch]{mlr3tuning::TunerBatch}} -> \code{\link[mlr3tuning:TunerBatchFromOptimizerBatch]{mlr3tuning::TunerBatchFromOptimizerBatch}} -> \code{TunerBatchHyperband}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-TunerBatchHyperband-new}{\code{TunerBatchHyperband$new()}}
\item \href{#method-TunerBatchHyperband-clone}{\code{TunerBatchHyperband$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="format"><a href='../../mlr3tuning/html/Tuner.html#method-Tuner-format'><code>mlr3tuning::Tuner$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="help"><a href='../../mlr3tuning/html/Tuner.html#method-Tuner-help'><code>mlr3tuning::Tuner$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3tuning" data-topic="Tuner" data-id="print"><a href='../../mlr3tuning/html/Tuner.html#method-Tuner-print'><code>mlr3tuning::Tuner$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3tuning" data-topic="TunerBatchFromOptimizerBatch" data-id="optimize"><a href='../../mlr3tuning/html/TunerBatchFromOptimizerBatch.html#method-TunerBatchFromOptimizerBatch-optimize'><code>mlr3tuning::TunerBatchFromOptimizerBatch$optimize()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TunerBatchHyperband-new"></a>}}
\if{latex}{\out{\hypertarget{method-TunerBatchHyperband-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TunerBatchHyperband$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TunerBatchHyperband-clone"></a>}}
\if{latex}{\out{\hypertarget{method-TunerBatchHyperband-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TunerBatchHyperband$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
