% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/miximp.R
\name{mixImp}
\alias{mixImp}
\title{Imputation for a mixture of continuous and categorical variables using
the general location model.}
\usage{
mixImp(
  obsData,
  nCat,
  M = 10,
  pd = FALSE,
  marginsType = 1,
  margins = NULL,
  designType = 1,
  design = NULL,
  steps = 100,
  rseed
)
}
\arguments{
\item{obsData}{The data frame to be imputed. The categorical variables must be
in the first \code{nCat} columns, and they must be coded using consecutive positive
integers.}

\item{nCat}{The number of categorical variables in \code{obsData}.}

\item{M}{Number of imputations to generate.}

\item{pd}{Specify whether to use posterior draws (\code{TRUE})
or not (\code{FALSE}).}

\item{marginsType}{An integer specifying what type of log-linear model to use for the
categorical variables. \code{marginsType=1}, the default, allows for all two-way associations
in the log-linear model. \code{marginsType=2} allows for all three-way associations (plus lower).
\code{marginsType=3} assumes a saturated log-linear model for the categorical variables.}

\item{margins}{If \code{marginsType} is not specified, \code{margins} must be
supplied to specify the margins of the log-linear model for the categorical variable.
See the help for \code{\link[mix]{ecm.mix}} for details on specifying \code{margins}.}

\item{designType}{An integer specifying how the continuous variables' means should depend
on the categorical variables. \code{designType=1}, the default, assumes the mean of each continuous
variable is a linear function with main effects of the categorical variables.
\code{designType=2} assumes each continuous variables has a separate mean for each
combination of the categorical variables.}

\item{design}{If \code{designType} is not specified, \code{design} must be supplied
to specify how the mean of the continuous variables depends on the categorical variables.
See the help for \code{\link[mix]{ecm.mix}} for details on specifying \code{design}.}

\item{steps}{If \code{pd} is \code{TRUE}, the \code{steps} argument specifies
how many MCMC iterations to perform.}

\item{rseed}{The value to set the \code{mix} package's random number seed to,
using the \code{rngseed} function of \code{mix}. This function must be called at least
once before imputing using \code{mix}. If the user wishes to set the seed using
\code{rngseed} before calling \code{mixImp}, set \code{rseed=NULL}.}
}
\value{
A list of imputed datasets, or if \code{M=1}, just the imputed data frame.
}
\description{
This function performs multiple imputation under a general location model
as described by Schafer (1997), using the \code{mix} package. Imputation can
either be performed using posterior draws (\code{pd=TRUE}) or conditonal on the maximum likelihood
estimate of the model parameters (\code{pd=FALSE}), referred to as maximum likelihood
multiple imputation by von Hippel (2018).
}
\details{
See the descriptions for \code{marginsType}, \code{margins}, \code{designType}, \code{design} and the documentation
in \code{\link[mix]{ecm.mix}} for details about how to specify the model.

Imputed datasets can be analysed using \code{\link{withinBetween}},
\code{\link{scoreBased}}, or for example the
\href{https://cran.r-project.org/package=bootImpute}{bootImpute} package.
}
\examples{
#simulate a partially observed dataset with a mixture of categorical and continuous variables
set.seed(1234)

n <- 100

#for simplicity we simulate completely independent categorical variables
x1 <- ceiling(3*runif(n))
x2 <- ceiling(2*runif(n))
x3 <- ceiling(2*runif(n))
y <- 1+0.5*(x1==2)+1.5*(x1==3)+x2+x3+rnorm(n)

temp <- data.frame(x1=x1,x2=x2,x3=x3,y=y)

#make some data missing in all variables
for (i in 1:4) {
  temp[(runif(n)<0.25),i] <- NA
}

#impute conditional on MLE, assuming two-way associations in the log-linear model
#and main effects of categorical variables on continuous one (the default)
imps <- mixImp(temp, nCat=3, M=10, pd=FALSE, rseed=4423)
}
\references{
Schafer J.L. (1997). Analysis of incomplete multivariate data.
Chapman & Hall, Boca Raton, Florida, USA.

von Hippel P.T. (2018) Maximum likelihood multiple imputation: faster,
more efficient imputation without posterior draws. \href{https://arxiv.org/abs/1210.0870v9}{arXiv:1210.0870v9}.
}
