% Generated by roxygen2 (4.0.2): do not edit by hand
\name{mixedMemModel}
\alias{mixedMemModel}
\title{Constructor for a Mixed Membership Model}
\usage{
mixedMemModel(Total, J, Rj, Nijr, K, Vj, alpha, theta, phi = NULL,
  delta = NULL, dist, obs)
}
\arguments{
\item{Total}{the number of individuals in the sample}

\item{J}{the number of variables observed on each individual}

\item{Rj}{vector of length J specifying the number of repeated measurements
on each variable}

\item{Nijr}{an array of dimension (Total, J, max(Rj)) indicating the number
of ranking levels for each replication. For multinomial and bernoulli
variables, Nijr[i,j,r] = 1. For rank variables, Nijr[i,j,r] indicates the
number of candidates ranked.}

\item{K}{the number of sub-populations}

\item{Vj}{vector of length J specifying the number of possible candidates
for each variable. For a bernoulli variable Vj[j] = 1.}

\item{alpha}{vector of length K which is the hyperparameter for Dirichlet
membership distribution}

\item{theta}{array of dimension (J,K,max(Vj)) which governs the variable
distributions. theta[j,k,] is the parameter for how sub-population k responds
to the variable j. Initially, any element of theta corresponding to a valid candidate/category must be positive (ie cannot be numerically 0).
However, if the number of candidates differs across variables, any unusued portions of theta should be 0.}

\item{phi}{array of dimension (Total,K) which specifies the variational
parameters for the membership vectors, lambda. If left NULL, it is initialized
to a uniformly across all groups. It is highly recommended to leave these NULL.}

\item{delta}{array of dimension (Total,J,max(Rj), max(N), K) which specifies
the variational parameters for the context vectors Z. If left blank, it is
 initialized to a uniformly across all sub-populations. It is highly recommended to leave these NULL.}

\item{dist}{vector of length J specifying variable types. Options are
"bernoulli", "multinomial" or "rank" corresponing to the distributions
 of the observed variables}

\item{obs}{an array of dimension (Total,J,max(Rj), max(N)) corresponding to
the observed data. For bernoulli random variables, this consists of 0/1's.
For multinomial or rank data this consists of integers 0,1,\ldots,(Vj[j]-1).}
}
\value{
The \code{mixedMemModel} object
}
\description{
Constructor for a \code{mixedMemModel} object which can be used for analysis
in the mixedMem package.
}
\details{
The function returns an object of \code{mixedMemModel} class. This object holds dimensions of the model,
the observed data, and the estimates of the model parameters. Once a \code{mixedMemModel} object is created,
it can be fit using the \code{mmVarFit} function. For additional details on usage, and model
assumptions, see the included vignette.
}
\examples{
set.seed(123)
Total <- 50 # 50 Individuals
J <- 3 # 3 different variables
# distributions of each variable
dist <- c("multinomial", "bernoulli", "rank")
# 100 repeated measures of the multinomial, 5 repeated measures of the
# bernoulli, 1 repeated measure of the rank
Rj <- c(100, 5, 1)

K <- 4 # 4 sub-populations
alpha <- rep(.5, K) #hyperparameter for dirichlet distribution

# Number of choices for each variable. Note the Bernoulli must have 1 choice
Vj <- c(10, 1, 4)


theta <- array(0, dim = c(J, K, max(Vj)))
# Parameters governing multinomial
theta[1,,] <- gtools::rdirichlet(K, rep(.3, Vj[1]))
#parameters governing bernoulli
theta[2,,] <- cbind(rbeta(K, 1,1), matrix(0, nrow = K, ncol = Vj[1]-1))
theta[3,,] <- cbind(gtools::rdirichlet(K, rep(.3, Vj[3])),
 matrix(0, nrow = K, ncol = Vj[1]-Vj[3]))

# Candidates selected for each observation. For Multinomial and Bernoulli, this is always 1
# For rank data, this will be the number of candidates ranked
Nijr = array(0, dim = c(Total, J, max(Rj)))
Nijr[,1,c(1:Rj[1])] = 1 # N_ijr is 1 for multinomial variables
Nijr[,2,c(1:Rj[2])] = 1 # N_ijr is 1 for Bernoulli variables
Nijr[,3,c(1:Rj[3])] = sample.int(Vj[3], size = Total, replace = TRUE)

# generate random sample of observations
sampleMixedMem <- rmixedMem(Total, J, Rj, Nijr, K, Vj,
dist, theta, alpha)

## Initialize a mixedMemModel object
test_model <- mixedMemModel(Total = Total, J = J,Rj = Rj,
 Nijr= Nijr, K = K, Vj = Vj,dist = dist, obs = sampleMixedMem$obs,
  alpha = alpha, theta = theta)
# Look at Summary of the initialized model
summary(test_model)
# Plot the current values for theta
plot(test_model)
}

