\name{mfr}
\alias{mfr}
\title{
Minimal free resolution.
}
\description{
The minimal free resolution of the edge ideal of a graph.
}
\usage{
mfr(g, verbose = FALSE, nocode = FALSE, seed = 42126)
}
\arguments{
  \item{g}{
a graph.
}
  \item{verbose}{
logical. Verbose output.
}
  \item{nocode}{
Do not run Singular.
}
  \item{seed}{
seed for the random number generator.
}
}
\details{
The minimal free resolution of a graph is an invariant defined by
a marriage of algebraic geometry and graph theory. If we notate
the vertices of a simple graph by variables, 
\eqn{x_1,\ldots,x_n}{x_1,...,x_n}, 
we can view the
edges as square-free quadratic monomials of the form \eqn{x_ix_j}.
The edge ideal of the graph is then the ideal of the polynomial ring
\eqn{R=k[x_1,\ldots,x_n]}{R=k[x_1,...,x_n]}
generated by these monomials. We will always assume
the field k is of characteristic 0.

Given an ideal of this type, one can uniquely define a minimal resolution
of the ideal. This is a sequence of rings and maps of the form:

\deqn{0 \rightarrow R^{\beta_m} \rightarrow R^{\beta_{m-1}}
\rightarrow \ldots \rightarrow R^{\beta_2} \rightarrow R \rightarrow R/I
\rightarrow 0}{0 --> R^{\beta_m} --> ... --> R^{\beta_2} --> R --> R/I --> 0}
The Betti numbers are the dimensions of the modules \eqn{\beta_i}.

There is a graded version of this as well, which keeps track of the
degrees of the generators. The projective dimension is the number
of non-trivial entries in the resolution, and the regularity is the
maximum degree of the graded resolution.

\code{mfr} computes the graded Betti numbers of the edge ideal of the graph.
This implements a recursive algorithm for the computation of the
Betti numbers that uses splitting edges: edges where the neighborhood
of one incident vertex contains the neighborhood of the other. If the
graph is chordal, this algorithm recurses; 
if \code{nocode} is FALSE and the graph is not chordal, it will call
Singular if it reaches a subgraph for which it does not know the 
resolution, and cannot find a splitting edge. 
The recursive algorithm is described in the reference.
If \code{nocode} is
TRUE and the graph is not chordal, instead of calling Singular it
chooses a likely edge (at random if \code{seed}>0, otherwise with
a heuristic) and procedes as if it were a splitting edge. In this
case the Betti numbers are unlikely to be correct, and should be
considered an approximation to the true Betti numbers. The returned
variable \code{punted} can be used to determine whether this is
the case or not.

An example of the graded Betti numbers returned from \code{mfr} 
is:

\tabular{rrrrrr}{
\tab [,1] \tab [,2] \tab [,3] \tab [,4] \tab [,5]\cr
	  [1,] \tab   1  \tab  0 \tab 0 \tab 0 \tab 0\cr
	  [2,] \tab   0  \tab  6 \tab 6 \tab 0 \tab 0\cr
	  [3,] \tab   0 \tab   0 \tab 3 \tab 6 \tab 2
}

\eqn{\beta_{1,1}} is always 1, and \eqn{\beta_{2,2}}
is the size of the graph. The above are the graded Betti
numbers for the cycle on 6 vertices.

The returned variable \code{punted} indicates whether the program
has attempted to call Singular and how many times. 
If \code{nocode=TRUE} was used, this
means that the returned resolution is approximate, and is unlikely
to be exactly correct. In this case, the larger this number
is, the more likely the Betti numbers will be incorrect and,
roughly, the larger the expected error.

Note that experiments have shown that the version of Singular 
that this code was tested on is unable to process
graphs with more than about 20 vertices, so a warning is printed to
the screen if Singular is called with a large enough graph. This warning
gives a wildly innaccurate estimate of how long the process is expected
to take, but if the number is in the thousands of minutes or more, it is
likely the process will not return before using up all memory and crashing
(or the heat-death of the universe, whichever comes first).

\code{mfr} does two things to help speed things up. First, it operates
on connected components independently, then combines the graded Betti
numbers appropriately. Even though the graph may be connected, the
recursive algorithm can disconnect the graph, and this has important
implications for speed, especially if the graph is not chordal and
Singular must ultimately be called. Secondly, \code{mfr} checks for
certain types of graphs (cycles, complete graphs, complete bipartite
graphs, stars) for which the Betti numbers are known and hence 
computable without resort to computationally complex algorithms.
}
\value{
\item{bettis }{total Betti numbers. First one is always 1, second is
the size of the graph.}
\item{graded }{graded Betti numbers}
\item{pd }{projective dimension}
\item{reg }{regularity}
\item{punted }{indicates whether Singular was called, or the Betti numbers
are approximate if \code{nocode} was TRUE}
}
\references{
Huy Tai Ha and Adam Van Tuyl, "Splittable ideals and the resolutions of monomial
ideals", J. Algebra, 309, 405-425, 2007.

Marchette and Johannsen, "Betti numbers of graphs", submitted for publication
in Statistical Analysis and Data Mining.
}
\author{
David Marchette, dmarchette@gmail.com
}
\seealso{
\code{\link{scanMFR}}, and \code{\link{MFRbwg}}, 
\code{\link{MFRComplete}},
\code{\link{MFRCompleteBipartite}},
\code{\link{MFRCycle}},
\code{\link{MFRStar}},
\code{\link{MFRPath}},
\code{\link{MFRWheel}} for special cases.
}

\examples{
g <- rg(10)
mfr(g)
}
\keyword{ math }
