% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rmev_wrapper.R
\name{rmev}
\alias{rmev}
\title{Exact simulations of multivariate extreme value distributions}
\usage{
rmev(n, d, param, asy, sigma, model = c("log", "alog", "neglog", "aneglog",
  "bilog", "negbilog", "hr", "xstud", "smith", "schlather", "ct", "dirmix"),
  alg = c("ef", "sm"), weights, vario, loc, grid = FALSE, ...)
}
\arguments{
\item{n}{number of observations}

\item{d}{dimension of sample}

\item{param}{parameter vector for the logistic, bilogistic, negative bilogistic and extremal Dirichlet (Coles and Tawn) model.
Parameter matrix for the Dirichlet mixture. Degree of freedoms for extremal student model.}

\item{asy}{list of asymmetry parameters, as in \code{\link[evd]{rmvevd}}, of \eqn{2^d-1} vectors of size corresponding to the power set of \code{d}, with sum to one constraints.}

\item{sigma}{covariance matrix for Husler-Reiss and extremal Student-t distributions}

\item{model}{choice between 1-parameter logistic and negative logistic, asymmetric logistic and negative logistic, bilogistic and the extremal Dirichlet model of Coles and Tawn,
the Brown-Resnick (which generate the Husler-Reiss MEV distribution), Smith, Schlather and extremal Student max-stable process , or the Dirichlet mixture.}

\item{alg}{algorithm, either simulation via extremal function or via the spectral measure. The extremal Dirichlet model is only implemented with \code{sm}.}

\item{weights}{vector of length \code{m} for the \code{m} mixture components. Must sum to one}

\item{vario}{function specifying the variogram. Used only if provided in conjonction with \code{loc} and if \code{sigma} is missing}

\item{loc}{\code{d} by \code{k} matrix of location, used as input in the variogram \code{vario} or as parameter for the Smith model. If \code{grid} is \code{TRUE}, unique entries should be supplied.}

\item{grid}{Logical. \code{TRUE} if the coordinates are two-dimensional grid points (spatial models).}

\item{...}{additional arguments for the \code{vario} function}
}
\value{
an \code{n} by \code{d} exact sample from the corresponding multivariate extreme value model
}
\description{
Implementation of the random number generators for multivariate extreme-value distributions
and max-stable processes based on the two algorithms described in
Dombry, Engelke and Oesting (2015).
}
\details{
The vector param differs depending on the model
\itemize{
 \item \code{log}: one dimensional parameter greater than 1
 \item \code{alog}: \eqn{2^d-d-1} dimensional parameter for \code{dep}. Values are recycled if needed.
 \item \code{neglog}: one dimensional positive parameter
 \item \code{aneglog}: \eqn{2^d-d-1} dimensional parameter for \code{dep}. Values are recycled if needed.
 \item \code{bilog}: \code{d}-dimensional vector of parameters in \eqn{[0,1]}
 \item \code{negbilog}: \code{d}-dimensional vector of negative parameters
 \item \code{ct}: \code{d}-dimensional vector of positive (a)symmetry parameters. Alternatively, a \eqn{d+1}
 vector consisting of the \code{d} Dirichlet parameters and the last entry is an index of regular variation in \eqn{(0, 1]} treated as scale
 \item \code{xstud}: one dimensional parameter corresponding to degrees of freedom \code{alpha}
 \item \code{dirmix}: \code{d} by \code{m}-dimensional matrix of positive (a)symmetry parameters
}

Stephenson points out that the multivariate asymmetric negative logistic model is not a valid distribution function.
The implementation in \code{mev} uses the same construction as the asymmetric logistic distribution,
and as such it does not match the bivariate implementation.
}
\examples{
set.seed(1)
rmev(n=100, d=3, param=2.5, model="log", alg="ef")
rmev(n=100, d=4, param=c(0.2,0.1,0.9,0.5), model="bilog", alg="sm")
## Spatial example using variogram, from Clement Dombry
#Variogram gamma(h) = scale*||h||^alpha
scale <- 0.5; alpha <- 1
vario <- function(x) scale*sqrt(sum(x^2))^alpha
#grid specification
grid.loc <- as.matrix(expand.grid(runif(4), runif(4)))
rmev(n=100, vario=vario,loc=grid.loc, model="hr")
#Example with a grid (generating an array)
rmev(n=10, sigma=cbind(c(2,1),c(1,3)), loc=cbind(runif(4),runif(4)),model="smith", grid=TRUE)
## Example with Dirichlet mixture
alpha.mat <- cbind(c(2,1,1),c(1,2,1),c(1,1,2))
rmev(n=100, param=alpha.mat, weights=rep(1/3,3), model="dirmix")
}
\author{
Leo Belzile
}
\references{
Dombry, Engelke and Oesting (2015). Exact simulation of max-stable processes, \emph{arXiv:1506.04430v1}, 1--24.
}
\seealso{
\link{rmevspec}, \link[evd]{rmvevd}, \link[evd]{rbvevd}
}

