% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fordyn_scenario.R
\name{fordyn_scenario}
\alias{fordyn_scenario}
\title{Scenario of forest dynamics}
\usage{
fordyn_scenario(
  sf,
  SpParams,
  meteo = NULL,
  management_scenario,
  volume_function = NULL,
  volume_arguments = NULL,
  local_control = defaultControl(),
  dispersal_control = default_dispersal_control(),
  dates = NULL,
  CO2ByYear = numeric(0),
  fire_regime = NULL,
  summary_function = NULL,
  summary_arguments = NULL,
  parallelize = FALSE,
  num_cores = detectCores() - 1,
  chunk_size = NULL,
  progress = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{sf}{An object of class \code{\link{sf}} with the following columns:
\itemize{
\item{\code{geometry}: Spatial geometry.}
\item{\code{id}: Stand identifiers.}
\item{\code{elevation}: Elevation above sea level (in m).}
\item{\code{slope}: Slope (in degrees).}
\item{\code{aspect}: Aspect (in degrees).}
\item{\code{forest}: Objects of class \code{\link{forest}}.}
\item{\code{soil}: Objects of class \code{\link{soil}}.}
\item{\code{state}: Objects of class \code{\link{spwbInput}} or \code{\link{growthInput}} (optional).}
\item{\code{meteo}: Data frames with weather data (required if parameter \code{meteo = NULL}).}
\item{\code{management_unit}: Management unit corresponding to each stand.}
\item{\code{represented_area_ha}: Area represented by each stand in hectares.}
\item{\code{ignition_weights}: Relative weights to determine stands to be burned. Optional, relevant when
\code{fire_regime} is supplied only).}
\item{\code{local_control}: A list of control parameters (optional). Used to override function parameter \code{local_control} for specific stands (values can be \code{NULL} for the remaining ones).}
}
Alternatively, the user may supply the result of a previous call to \code{fordyn_scenario}, where
to continue simulations.}

\item{SpParams}{A data frame with species parameters (see \code{\link{SpParamsMED}}).}

\item{meteo}{Meteorology data (see \code{\link{fordyn_spatial}}).}

\item{management_scenario}{A list defining the management scenario (see \code{\link{create_management_scenario}})}

\item{volume_function}{A function accepting a forest object or a tree data table, and a species parameter table, as input and
returning the wood volume (m3/ha) corresponding to each tree cohort. The function may accept additional arguments.
If NULL, the default volume function is used (not recommended!).}

\item{volume_arguments}{List with additional arguments for the volume function.}

\item{local_control}{A list of local model control parameters (see \code{\link{defaultControl}}).}

\item{dispersal_control}{A list of dispersal control parameters (see \code{\link{default_dispersal_control}}). If NULL, then dispersal is not simulated.}

\item{dates}{A \code{\link{Date}} object with the days of the period to be simulated. If \code{NULL}, then the whole period of \code{meteo} is used.}

\item{CO2ByYear}{A named numeric vector with years as names and atmospheric CO2 concentration (in ppm) as values. Used to specify annual changes in CO2 concentration along the simulation (as an alternative to specifying daily values in \code{meteo}).}

\item{fire_regime}{A list of parameters defining the fire regime (see \code{\link{create_fire_regime}}) or
a matrix representing a fire regime instance (see \code{\link{fire_regime_instance}}).
If NULL, wildfires are not simulated. Details are given in \code{\link{fordyn_spatial}}.}

\item{summary_function}{An appropriate function to calculate summaries from an object of class 'fordyn' (e.g., \code{\link{summary.fordyn}}).}

\item{summary_arguments}{List with additional arguments for the summary function.}

\item{parallelize}{Boolean flag to try parallelization (will use all clusters minus one).}

\item{num_cores}{Integer with the number of cores to be used for parallel computation.}

\item{chunk_size}{Integer indicating the size of chunks to be sent to different processes (by default, the number of spatial elements divided by the number of cores).}

\item{progress}{Boolean flag to display progress information for simulations.}

\item{verbose}{Boolean flag to display additional console output.}
}
\value{
An list of class 'fordyn_scenario' with the following elements:
\itemize{
\item{\code{result_sf}: An object of class 'sf' using a UTM projection and containing four elements:
\itemize{
\item{\code{geometry}: Spatial geometry.}
\item{\code{id}: Stand id, taken from the input.}
\item{\code{tree_table}: A list of data frames for each simulated stand, containing the living trees at each time step.}
\item{\code{shrub_table}: A list of data frames for each simulated stand, containing the living shrub at each time step.}
\item{\code{dead_tree_table}: A list of data frames for each simulated stand, containing the dead trees at each time step.}
\item{\code{dead_shrub_table}: A list of data frames for each simulated stand, containing the dead shrub at each time step.}
\item{\code{cut_tree_table}: A list of data frames for each simulated stand, containing the cut trees at each time step.}
\item{\code{cut_shrub_table}: A list of data frames for each simulated stand, containing the cut shrub at each time step.}
\item{\code{summary}: A list of model output summaries for each simulated stand (if \code{summary_function} was not \code{NULL}).}
}
}
\item{\code{result_volumes}: A data frame with initial, growth, extracted and final volumes (m3) by year. In demand-based scenarios volumes corresponding to species with demand are also included.}
\item{\code{result_volumes_spp}: A data frame with growth and extracted volumes (m3) by species and year.}
\item{\code{result_volumes_demand}: In demand-based scenarios target volumes are also included, a data frame with growth, target and extracted volumes (m3) by demand entity and year. .}
\item{\code{next_sf}: An object of class 'sf' to continue simulations in subsequent calls to \code{fordyn_scenario}.}
\item{\code{next_demand}: In demand-based scenarios, a list with information (i.e. demand offset by species and last volume growth)
to modify demand in subsequent calls to \code{fordyn_scenario}.}
}
}
\description{
Evaluates forest dynamics over a landscape including climate and management scenarios
}
\details{
This function allows coordinating the dynamics of simulated forest stands via a management scenario
defined at the landscape/regional level (see different kinds of scenarios and how to specify them in \code{\link{create_management_scenario}}).

The input 'sf' object has to be in a Universal Transverse Mercator (UTM) coordinate system (or any other projection using meters as length unit)
for appropriate behavior of dispersal sub-model.

For each year to be simulated, the function determines which forest stands will be managed, possibly depending on the demand,
and then calls function \code{\link{fordyn_spatial}} for one year (normally including parallelization).
If the simulation of some stands results in an error, the function will try to restore
the previous state of the forest stand for the next year steps. Finally, the function evaluates how much of the specified demand
has been fulfilled and stores the results, including demand offsets to be applied the year after.

Management is implemented using the \code{\link{defaultManagementFunction}} in medfate,
meaning that management parameters need to follow the structure of \code{\link{defaultManagementArguments}}

Details about the inclusion of fire regimes in simulations are explained in \code{\link{fordyn_spatial}}.
}
\examples{
\donttest{
# Load example landscape data
data("example_ifn")
  
# Load example meteo data frame from package meteoland
data("examplemeteo")
  
# Load default medfate parameters
data("SpParamsMED")

# Creates scenario with one management unit and annual demand for P. nigra 
scen <- create_management_scenario(1, c("Pinus nigra/Pinus sylvestris" = 2300))

# Assign management unit to all stands
example_ifn$management_unit <- 1 

# Assume that each stand represents 1km2 = 100 ha
example_ifn$represented_area_ha <- 100

# Transform to UTM31 (necessary for dispersal)
example_ifn_utm31 <- sf::st_transform(example_ifn, crs = 32631)

# Subset three plots to speed-up calculations
example_subset <- example_ifn_utm31[31:33, ]

# Launch simulation scenario
fs_12 <- fordyn_scenario(example_subset, SpParamsMED, meteo = examplemeteo, 
                         volume_function = NULL, management_scenario = scen,
                         parallelize = FALSE)
}
}
\seealso{
\code{\link{fordyn_spatial}}, \code{\link{create_management_scenario}}, \code{\link{dispersal}}
}
\author{
Miquel De \enc{Cáceres}{Caceres} Ainsa, CREAF

Aitor \enc{Améztegui}{Ameztegui}, UdL
}
