% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make_play.R
\name{make_play}
\alias{make_play}
\alias{play_diffusion}
\alias{play_diffusions}
\title{Making diffusion models on networks}
\usage{
play_diffusion(
  .data,
  seeds = 1,
  contact = NULL,
  prevalence = 0,
  thresholds = 1,
  transmissibility = 1,
  latency = 0,
  recovery = 0,
  waning = 0,
  immune = NULL,
  steps
)

play_diffusions(
  .data,
  seeds = 1,
  contact = NULL,
  prevalence = 0,
  thresholds = 1,
  transmissibility = 1,
  latency = 0,
  recovery = 0,
  waning = 0,
  immune = NULL,
  steps,
  times = 5,
  strategy = "sequential",
  verbose = FALSE
)
}
\arguments{
\item{.data}{An object of a manynet-consistent class:
\itemize{
\item matrix (adjacency or incidence) from \code{{base}} R
\item edgelist, a data frame from \code{{base}} R or tibble from \code{{tibble}}
\item igraph, from the \code{{igraph}} package
\item network, from the \code{{network}} package
\item tbl_graph, from the \code{{tidygraph}} package
}}

\item{seeds}{A valid mark vector the length of the
number of nodes in the network.}

\item{contact}{A matrix or network that replaces ".data" with some
other explicit contact network, e.g.
\code{create_components(.data, membership = node_in_structural(.data))}.
Can be of arbitrary complexity, but must of the same dimensions
as .data.}

\item{prevalence}{The proportion that global prevalence contributes
to diffusion.
That is, if prevalence is 0.5, then the current number of infections
is multiplied by 0.5 and added
"prevalence" is 0 by default, i.e. there is no global mechanism.
Note that this is endogenously defined and is updated
at the outset of each step.}

\item{thresholds}{A numeric vector indicating the thresholds
each node has. By default 1.
A single number means a generic threshold;
for thresholds that vary among the population please use a vector
the length of the number of nodes in the network.
If 1 or larger, the threshold is interpreted as a simple count
of the number of contacts/exposures sufficient for infection.
If less than 1, the threshold is interpreted as complex,
where the threshold concerns the proportion of contacts.}

\item{transmissibility}{The transmission rate probability,
\eqn{\beta}.
By default 1, which means any node for which the threshold is met
or exceeded will become infected.
Anything lower means a correspondingly lower probability of adoption,
even when the threshold is met or exceeded.}

\item{latency}{The inverse probability those who have been exposed
become infectious (infected), \eqn{\sigma} or \eqn{\kappa}.
For example, if exposed individuals take, on average,
four days to become infectious, then \eqn{\sigma = 0.75} (1/1-0.75 = 1/0.25 = 4).
By default 0, which means those exposed become immediately infectious (i.e. an SI model).
Anything higher results in e.g. a SEI model.}

\item{recovery}{The probability those who are infected
recover, \eqn{\gamma}.
For example, if infected individuals take, on average,
four days to recover, then \eqn{\gamma = 0.25}.
By default 0, which means there is no recovery (i.e. an SI model).
Anything higher results in an SIR model.}

\item{waning}{The probability those who are recovered
become susceptible again, \eqn{\xi}.
For example, if recovered individuals take, on average,
four days to lose their immunity, then \eqn{\xi = 0.25}.
By default 0, which means any recovered individuals retain lifelong immunity (i.e. an SIR model).
Anything higher results in e.g. a SIRS model.
\eqn{\xi = 1} would mean there is no period of immunity, e.g. an SIS model.}

\item{immune}{A logical or numeric vector identifying nodes
that begin the diffusion process as already recovered.
This could be interpreted as those who are vaccinated or equivalent.
Note however that a waning parameter will affect these nodes too.
By default NULL, indicating that no nodes begin immune.}

\item{steps}{The number of steps forward in the diffusion to play.
By default the number of nodes in the network.
If \code{steps = Inf} then the diffusion process will continue until
there are no new infections or all nodes are infected.}

\item{times}{Integer indicating number of simulations.
By default \code{times=5}, but 1,000 - 10,000 simulations recommended for publication-ready results.}

\item{strategy}{If \code{{furrr}} is installed, then multiple cores can be used to accelerate the simulations.
By default "sequential", but if multiple cores available, then "multisession" or "multicore" may be useful.
Generally this is useful only when times > 1000. See \code{{furrr}} for more.}

\item{verbose}{Whether the function should report on its progress.
By default FALSE. See \code{{progressr}} for more.}
}
\description{
These functions simulate diffusion or compartment models upon a network.
\itemize{
\item \code{play_diffusion()} runs a single simulation of a compartment model,
allowing the results to be visualised and examined.
\item \code{play_diffusions()} runs multiple simulations of a compartment model
for more robust inference.
}

These functions allow both a full range of compartment models,
as well as simplex and complex diffusion to be simulated upon a network.
}
\section{Simple and complex diffusion}{


By default, the function will simulate a simple diffusion process in
which some infectious disease or idea diffuses from seeds through
contacts at some constant rate (\code{transmissibility}).

These \code{seeds} can be specified by a vector index
(the number of the position of each node in the network that should serve as a seed)
or as a logical vector where TRUE is interpreted as already infected.

\code{thresholds} can be set such that adoption/infection requires more than one
(the default) contact already being infected.
This parameter also accepts a vector so that thresholds can vary.

Complex diffusion is where the \code{thresholds} are defined less than one.
In this case, the thresholds are interpreted as proportional.
That is, the threshold to adoption/infection is defined by the
proportion of the node's contacts infected.

Nodes that cannot be infected can be indicated as \code{immune}
with a logical vector or index, similar to how \code{seeds} are identified.
Note that \code{immune} nodes are interpreted internally as Recovered (R)
and are thus subject to \code{waning} (see below).
}

\section{Compartment models}{


Compartment models are flexible models of diffusion or contagion,
where nodes are compartmentalised into one of two or more categories.

The most basic model is the SI model.
The SI model is the default in \code{play_diffusion()}/\code{play_diffusions()},
where nodes can only move from the Susceptible (S) category to the
Infected (I) category.
Whether nodes move from S to I depends on whether they are exposed
to the infection, for instance through a contact,
the \code{transmissibility} of the disease,
and their \code{thresholds} to the disease.

Another common model is the SIR model.
Here nodes move from S to I, as above, but additionally they can
move from I to a Recovered (R) status.
The probability that an infected node recovers at a timepoint
is controlled by the \code{recovery} parameter.

The next most common models are the SIS and SIRS models.
Here nodes move from S to I or additionally to R, as above,
but additionally they can move from I or R back to a Susceptible (S) state.
This probability is governed by the \code{waning} parameter.
Where \code{recover > 0} and \code{waning = 1}, the Recovery (R) state will be skipped
and the node will return immediately to the Susceptible (S) compartment.

Lastly, these functions also offer the possibility of specifying
a latency period in which nodes have been infected but are not yet infectious.
Where \code{latency > 0}, an additional Exposed (E) compartment is introduced
that governs the probability that a node moves from this E compartment
to infectiousness (I).
This can be used in in SEI, SEIS, SEIR, and SEIRS models.
}

\examples{
  smeg <- generate_smallworld(15, 0.025)
  plot(play_diffusion(smeg, recovery = 0.4))
  #graphr(play_diffusion(ison_karateka))
  plot(play_diffusions(smeg, times = 10))
}
\seealso{
Other makes: 
\code{\link{make_cran}},
\code{\link{make_create}},
\code{\link{make_explicit}},
\code{\link{make_learning}},
\code{\link{make_random}},
\code{\link{make_read}},
\code{\link{make_stochastic}},
\code{\link{make_write}}

Other models: 
\code{\link{make_learning}}

Other diffusion: 
\code{\link{measure_diffusion_infection}},
\code{\link{measure_diffusion_net}},
\code{\link{measure_diffusion_node}},
\code{\link{member_diffusion}}
}
\concept{diffusion}
\concept{makes}
\concept{models}
