% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pooling-basic-funs.R
\name{pooling_mc}
\alias{pooling_mc}
\title{Monte Carlo Simulation for Estimating the Number of Assays Required
when Using Pooled Testing}
\usage{
pooling_mc(v, s = NULL, K = 5, vf_cut = 1000, lod = 0,
  method = "mmpa", perm_num = 100, msg = F)
}
\arguments{
\item{v}{A vector of non-negative numerical assay results.}

\item{s}{A vector of risk scores; \code{s} must be provided if \code{method = "mmpa"}
and have the same length as \code{v}. The risk score \code{s} needs to be positively
associated with \code{v}; othewise an error message will be generated.}

\item{K}{Pool size; default is \code{K = 5}.}

\item{vf_cut}{Cutoff value for defining positive cases;
default is \code{vf_cut = 1000}.}

\item{lod}{A vector of lower limits of detection or a scalar if the limits are the
same; default is \code{lod = 0}.}

\item{method}{Method that is used for pooled testing; must be one of \code{minipool},
\code{mpa}, and \code{mmpa}. By default, \code{method = "mmpa"}.}

\item{perm_num}{The number of permutation to be used for the calculation;
default is \code{100}.}

\item{msg}{Message generated during calculation; default is \code{FALSE}.}
}
\value{
The outcome is a matrix of dimension \code{num_pool} by \code{perm_num}.
The row number is the number of pools (\code{num_pool}) from each permutation
of the data, which is
determined by the sample size \code{N} and pool size \code{K}; \code{num_pool
= N\%/\%K}. The column number is the number of
permutations (\code{num_pool}).
}
\description{
This function uses Monte Carlo (MC) to simulate different orders in
which the samples would be collected to form pools. Unlike the
function \code{minipool}, \code{mpa}, and \code{mmpa} that calculate
the number of assays
needed for pools that are formed following the exact order
of the samples that are listed in the data, this function
\code{pooling_mc} permutes the data many (\code{perm_num}) times
so as to estimate the average number of
assays required (ATR) per individual. Using MC avoids the dependence
on any
specific ordering of forming pools.
}
\examples{
### sample size = 300
n = 300;
set.seed(100)
pvl = rgamma(n, shape = 2.8, scale = 150)
summary(pvl)
#   Min. 1st Qu.  Median    Mean 3rd Qu.    Max.
#   53      225      392     424    564    1373
riskscore = (rank(pvl)/n) * 0.5 + runif(n) * 0.5
cor(pvl, riskscore, method = "spearman")
#   [1] 0.69
### Pool size K is set to 5
K=5;
### so, the number of pools = 60
n.pool  = n/K; n.pool
#   [1] 60
foo = pooling_mc(pvl, riskscore, perm_num = 100)
### Average number of assays needed per pool for each of the 100
### permutations of the data
apply(foo, 2, mean)
#   [1] 3.43 3.33 3.35 3.47 3.37 3.33 3.37 3.27 3.43 3.28 3.32 3.35 3.35 3.37
#   [15] 3.38 3.37 3.30 3.43 3.28 3.38 3.42 3.35 3.35 3.48 3.30 3.47 3.40 3.35
#  [29] 3.25 3.30 3.38 3.43 3.25 3.45 3.35 3.33 3.42 3.38 3.40 3.33 3.32 3.38
#   [43] 3.33 3.37 3.37 3.33 3.35 3.38 3.38 3.30 3.30 3.33 3.37 3.32 3.30 3.40
#   [57] 3.37 3.42 3.30 3.37 3.38 3.32 3.45 3.38 3.37 3.50 3.33 3.40 3.28 3.37
#   [71] 3.23 3.33 3.23 3.42 3.32 3.32 3.45 3.35 3.32 3.32 3.33 3.33 3.30 3.38
#   [85] 3.37 3.33 3.33 3.20 3.37 3.33 3.30 3.40 3.40 3.32 3.33 3.37 3.40 3.38
#   [99] 3.30 3.33
### Estimated average number of assays needed per pool
mean(foo)
#   3.35
### Estimated average number of assays needed per individual
mean(foo)/K
#   [1] 0.67
}
\references{
Bilder CR, Tebbs JM, Chen P. Informative retesting. Journal of the American
Statistical Association. 2010;105(491):942-955.

May S, Gamst A, Haubrich R, Benson C, Smith DM. Pooled nucleic acid testing to
identify antiretroviral treatment failure during HIV infection. Journal of Acquired
Immune Deficiency Syndromes. 2010;53(2):194-201.

Dorfman R. The detection of defective members of large populations. The
Annals of Mathematical Statistics. 1943;14(4):436-440.

Our manuscript; to be added.
}
\seealso{
\link{minipool}, \link{mpa}, \link{mmpa}
}
\keyword{Pooling.}
