% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linkGRASS.R
\name{linkGRASS}
\alias{linkGRASS}
\title{Locate and set up 'GRASS' API bindings}
\usage{
linkGRASS(
  x = NULL,
  default_GRASS = NULL,
  search_path = NULL,
  ver_select = FALSE,
  gisdbase_exist = FALSE,
  gisdbase = NULL,
  use_home = FALSE,
  location = NULL,
  spatial_params = NULL,
  resolution = NULL,
  quiet = TRUE,
  returnPaths = TRUE
)
}
\arguments{
\item{x}{raster/terra or sf/sp object}

\item{default_GRASS}{default is \code{NULL} If is \code{NULL} an automatic search for all installed versions is performed. 
If you provide a valid list the corresponding version is initialized. An example for OSGeo4W64 is:
\code{c("C:/OSGeo4W64","grass-7.0.5","osgeo4w")}}

\item{search_path}{path or mounting point that will be searched}

\item{ver_select}{boolean if TRUE you may choose interactively the binary version (if found  more than one),  by default FALSE}

\item{gisdbase_exist}{default is FALSE if set to TRUE the arguments gisdbase and location are expected to be an existing GRASS gisdbase}

\item{gisdbase}{default is \code{NULL}, invoke \code{tempdir()} to the 'GRASS' database. Alternatively you can provide a individual path.}

\item{use_home}{default is \code{FALSE}, set the GISRC path to tempdir(), if TRUE the HOME or USERPROFILE setting is used for writing the GISRC file}

\item{location}{default is \code{NULL}, invoke \code{basename(tempfile())} for defining the 'GRASS' location. Alternatively you can provide a individual path.}

\item{spatial_params}{default is \code{NULL}. Instead of a spatial object you may provide the geometry as a list. E.g. c(xmin,ymin,xmax,ymax,proj4_string)}

\item{resolution}{resolution in map units for the GRASS raster cells}

\item{quiet}{boolean  switch for supressing console messages default is TRUE}

\item{returnPaths}{boolean if set to FALSE the pathes of the selected version are written 
to the PATH variable only, otherwise all paths and versions of the installed GRASS versions ae returned.}
}
\description{
Initializes the session environment and the system paths for an easy access to
\href{https://grass.osgeo.org/}{'GRASS GIS 7.x/8.x'}.  The correct setup of the spatial and
 projection parameters is automatically performed by using either an existing and valid 
 \code{raster}, \code{terra}, \code{sp} or \code{sf} object, 
 or manually by providing a list containing the minimum parameters needed.\cr
}
\details{
The concept is straightforward but for an all days usage helpful. Either you need to 
provide a \code{raster} or \code{sp} \code{sf} spatial object
 which has correct spatial and projection properties or you may link directlxy to an existing 'GRASS' gisdbase and mapset. 
 If you choose an spatial object to initialize a correct 'GRASS' mapset it is used to create either a temporary or a permanent 
 \href{https://CRAN.R-project.org/package=rgrass}{rgrass} environment including the correct 'GRASS ' structure.\cr\cr
 The most time consuming part on 'Windows' Systems is the search process. This can easily take 10 or more minutes. 
 To speed up this process you can also provide a correct parameter set. Best way to do so is to call \code{searchGRASSW} or for 'Linux' \code{searchGRASSX} manually. 
 and call \code{linkGRASS} with the version arguments of your choice. linkGRASS initializes the usage of GRASS7.
}
\note{
'GRASS GIS' is excellently supported by the
 \code{rgrass} wrapper package. Nevertheless 'GRASS GIS' is well known for
 its high demands regarding the correct spatial and reference setup  
 of work space and environment requirements. This becomes even worse on 'Windows' 
 platforms or if several alternative 'GRASS GIS' installations are available.
 If one knows what to do the \code{rgrass} package setup function 
 \code{rgrass::initGRASS} works fine under Linux. 
 This is also valid for well known configurations under the 'Windows' operation system. 
 Nevertheless on university labs or on company computers with restricted privileges and/or using different releases
 like the  \href{https://trac.osgeo.org/osgeo4w/}{'OSGeo4W'} distribution and the  
 \href{https://grass.osgeo.org/download/windows/}{'GRASS ' stand-alone} installation, 
 or different software releases (e.g. 'GRASS 7.0.5 and GRASS 8.1.0), it becomes often cumbersome or even 
 impossible to get the correct linkages. \cr 
 The function \code{linkGRASS} tries to find all valid 'GRASS GIS' binaries by analyzing
 the startup script files of 'GRASS GIS'. After identifying the 'GRASS GIS' binaries all
 necessary system variables and settings will be generated and passed to a temporary R enviroment.

If you have more than one valid installation and run \code{linkGRASS()} without arguments, you will be ask to select one.
}
\examples{

run = FALSE
if (run) {
library(link2GI)
require(sf)

# proj folders
projRootDir = tempdir()
paths = link2GI::initProj(projRootDir = projRootDir,
                         projFolders = c("project1/"))
                         
# get  data                         
nc = st_read(system.file("shape/nc.shp", package="sf"))

# Automatic search and find of GRASS binaries 
# using the nc sf data object for spatial referencing
# This is the highly recommended linking procedure for on the fly jobs
# NOTE: if more than one GRASS installation is found you have to choose. 
grass = linkGRASS(nc,returnPaths = TRUE)
if (grass$exist){

# CREATE and link to a permanent GRASS folder at "projRootDir", location named "project1" 
linkGRASS(nc, gisdbase = projRootDir, location = "project1")   

# ONLY LINK to a permanent GRASS folder at "projRootDir", location named "project1" 
linkGRASS(gisdbase = projRootDir, location = "project1", gisdbase_exist = TRUE )   


# setting up GRASS manually with spatial parameters of the nc data
proj4_string = as.character(sp::CRS("+init=epsg:28992"))
linkGRASS(spatial_params = c(178605,329714,181390,333611,proj4_string)) 

# creating a GRASS gisdbase manually with spatial parameters of the nc data 
# additionally using a peramanent directory "projRootDir" and the location "nc_spatial_params "
proj4_string = as.character(sp::CRS("+init=epsg:4267"))
linkGRASS(gisdbase = projRootDir,
           location = "nc_spatial_params",
           spatial_params = c(-84.32385, 33.88199,-75.45698,36.58965,proj4_string))
}

## Some more examples related to interactive selection or OS specific settings

# SELECT the GRASS installation and define the search location
linkGRASS(nc, ver_select = TRUE, search_path = "~")

# SELECT the GRASS installation 
linkGRASS(nc, ver_select = TRUE)

# Typical osge4W installation (QGIS), using the meuse sp data object for spatial referencing
linkGRASS(nc,c("C:/Program Files/QGIS 2.18","grass-7.2.1","osgeo4W")) 

# Typical osgeo4W installation (rootdir), using the meuse sp data object for spatial referencing 
linkGRASS(nc,c("C:/OSGeo4W64/","grass-7.2.2","osgeo4W"))

}

}
\author{
Chris Reudenbach
}
