% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lineqGP.R
\name{simulate.lineqGP}
\alias{simulate.lineqGP}
\title{Simulation Method for the \code{"lineqGP"} S3 Class}
\usage{
\method{simulate}{lineqGP}(object, nsim = 1, seed = NULL, xtest, ...)
}
\arguments{
\item{object}{an object with class \code{"lineqGP"}.}

\item{nsim}{the number of simulations.}

\item{seed}{see \code{\link{simulate}}.}

\item{xtest}{a vector (or matrix) with the test input design.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
A \code{"lineqGP"} object with the following elements.
\item{x}{a vector (or matrix) with the training input design.}
\item{y}{the training output vector at \code{x}.}
\item{xtest}{a vector (or matrix) with the test input design.}
\item{Phi.test}{a matrix corresponding to the hat basis functions evaluated
at \code{xtest}. The basis functions are indexed by rows.}
\item{xi.sim}{the posterior sample-path of the finite-dimensional Gaussian vector.}
\item{ysim}{the posterior sample-path of the observed GP.
Note: \code{ysim = Phi.test \%*\% xi.sim}.}
}
\description{
Simulation method for the \code{"lineqGP"} S3 class.
}
\details{
The posterior sample-path of the finite-dimensional GP with linear inequality
constraints are computed. Here, \eqn{\boldsymbol{\xi}}{\xi} is a centred Gaussian
vector with covariance \eqn{\boldsymbol{\Gamma}}{\Gamma}, s.t.
\eqn{\boldsymbol{\Phi} \boldsymbol{\xi} = \boldsymbol{y}}{\Phi \xi = y}
(interpolation constraints) and
\eqn{\boldsymbol{l} \leq \boldsymbol{\Lambda} \boldsymbol{\xi} \leq \boldsymbol{u}}{lb \le \Lambda \xi \le ub}
(inequality constraints).
}
\examples{
# creating the model
sigfun <- function(x) return(1/(1+exp(-7*(x-0.5))))
x <- seq(0, 1, length = 5)
y <- sigfun(x)
model <- create(class = "lineqGP", x, y, constrType = "monotonicity")

# updating and expanding the model
model$localParam$m <- 30
model$kernParam$par <- c(1, 0.2)
model <- augment(model)

# sampling from the model
xtest <- seq(0, 1, length = 100)
ytest <- sigfun(xtest)
sim.model <- simulate(model, nsim = 50, seed = 1, xtest = xtest)
mu <- apply(sim.model$ysim, 1, mean)
qtls <- apply(sim.model$ysim, 1, quantile, probs = c(0.05, 0.95))
matplot(xtest, t(qtls), type = "l", lty = 1, col = "gray90",
        main = "Constrained Kriging model")
polygon(c(xtest, rev(xtest)), c(qtls[2,], rev(qtls[1,])), col = "gray90", border = NA)
lines(xtest, ytest, lty = 2)
lines(xtest, mu, type = "l", col = "darkgreen")
points(x, y, pch = 20)
legend("right", c("ytrain", "ytest", "mean", "confidence"), lty = c(NaN,2,1,NaN),
       pch = c(20,NaN,NaN,15), col = c("black", "black", "darkgreen", "gray80"))

}
\references{
Lopez-Lopera, A. F., Bachoc, F., Durrande, N., and Roustant, O. (2017),
"Finite-dimensional Gaussian approximation with linear inequality constraints".
\emph{ArXiv e-prints}
\href{https://arxiv.org/abs/1710.07453}{[link]}
}
\seealso{
\code{\link{create.lineqGP}}, \code{\link{augment.lineqGP}},
         \code{\link{predict.lineqGP}}
}
\author{
A. F. Lopez-Lopera.
}
