% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lineqGP.R
\name{predict.lineqGP}
\alias{predict.lineqGP}
\title{Prediction Method for the \code{"lineqGP"} S3 Class}
\usage{
\method{predict}{lineqGP}(object, xtest, ...)
}
\arguments{
\item{object}{an object with class \code{"lineqGP"}.}

\item{xtest}{a vector (or matrix) with the test input design.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
A \code{"lineqGP"} object with the following elements.
\item{Lambda}{a matrix corresponding to the linear set of inequality constraints.}
\item{lb}{the lower bound vector of the inequalities constraints.}
\item{ub}{the upper bound vector of the inequalities constraints.}
\item{Phi.test}{a matrix corresponding to the hat basis functions evaluated
at \code{xtest}. The basis functions are indexed by rows.}
\item{mu}{the unconstrained GP mean predictor.}
\item{Sigma}{the unconstrained GP prediction conditional covariance matrix.}
\item{xi.map}{the GP maximum a posteriori (MAP) predictor given the inequality constraints.}
}
\description{
Prediction method for the \code{"lineqGP"} S3 class.
}
\details{
The posterior paramaters of the finite-dimensional GP with linear inequality
constraints are computed. Here, \eqn{\boldsymbol{\xi}}{\xi} is a centred Gaussian
vector with covariance \eqn{\boldsymbol{\Gamma}}{\Gamma}, s.t.
\eqn{\boldsymbol{\Phi} \boldsymbol{\xi} = \boldsymbol{y}}{\Phi \xi = y}
(interpolation constraints) and
\eqn{\boldsymbol{l} \leq \boldsymbol{\Lambda} \boldsymbol{\xi} \leq \boldsymbol{u}}{lb \le \Lambda \xi \le ub}
(inequality constraints).
}
\examples{
# creating the model
sigfun <- function(x) return(1/(1+exp(-7*(x-0.5))))
x <- seq(0, 1, length = 5)
y <- sigfun(x)
model <- create(class = "lineqGP", x, y, constrType = "monotonicity")

# updating and expanding the model
model$localParam$m <- 30
model$kernParam$par <- c(1, 0.2)
model <- augment(model)

# predictions from the model
xtest <- seq(0, 1, length = 100)
ytest <- sigfun(xtest)
pred <- predict(model, xtest)
plot(xtest, ytest, type = "l", lty = 2, main = "Kriging predictions")
lines(xtest, pred$Phi.test \%*\% pred$mu, type = "l", col = "blue")
lines(xtest, pred$Phi.test \%*\% pred$xi.map, type = "l", col = "red")
legend("right", c("ytest", "mean", "mode"), lty = c(2,1,1),
       col = c("black", "blue", "red"))

}
\references{
Lopez-Lopera, A. F., Bachoc, F., Durrande, N., and Roustant, O. (2017),
"Finite-dimensional Gaussian approximation with linear inequality constraints".
\emph{ArXiv e-prints}
\href{https://arxiv.org/abs/1710.07453}{[link]}
}
\seealso{
\code{\link{create.lineqGP}}, \code{\link{augment.lineqGP}},
         \code{\link{simulate.lineqGP}}
}
\author{
A. F. Lopez-Lopera.
}
