% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matrixcreation.R
\name{fleslie}
\alias{fleslie}
\title{Create Function-based Age-based (Leslie) Matrix Projection Model}
\usage{
fleslie(
  year = "all",
  patch = "all",
  prebreeding = TRUE,
  data = NULL,
  modelsuite = NULL,
  surv_model = NULL,
  fec_model = NULL,
  paramnames = NULL,
  start_age = NA,
  last_age = NA,
  fecage_min = NA,
  fecage_max = NA,
  continue = TRUE,
  inda = NULL,
  indb = NULL,
  indc = NULL,
  surv_dev = 0,
  fec_dev = 0,
  density = NA,
  repmod = 1,
  random.inda = FALSE,
  random.indb = FALSE,
  random.indc = FALSE,
  negfec = FALSE,
  exp_tol = 700,
  theta_tol = 1e+08
)
}
\arguments{
\item{year}{A variable corresponding to observation occasion, or a set
of such values, given in values associated with the year term used in linear 
model development. Defaults to \code{"all"}, in which case matrices will be
estimated for all occasions.}

\item{patch}{A variable designating which patches or subpopulations will have
matrices estimated. Defaults to \code{"all"}, but can also be set to specific
patch names or a vector thereof.}

\item{prebreeding}{A logical value indicating whether the life history model
is a pre-breeding model. Defaults to \code{TRUE}.}

\item{data}{The historical vertical demographic data frame used to estimate
vital rates (class \code{hfvdata}). The original data frame is generally
required in order to initialize occasions and patches properly, and to assess
the range of ages observed in the population. Not required if option
\code{modelsuite} is set to a \code{vrm_input} object.}

\item{modelsuite}{One of two optional lists. THe first is an optional
\code{lefkoMod} object holding the vital rate models. Alternatively,
an object of class \code{vrm_input} may be provided. If given, then
\code{surv_model}, \code{fec_model}, and \code{paramnames} are not required.
No models should include size or reproductive status in any occasion, nor
should they include any variable for occasion \emph{t}-1. Note that the
modelsuite must have been created from a \code{modelsearch()} run in which
\code{vitalrates = c("surv", "fec")} and the \code{suite} option was set to
either \code{age} or \code{cons}.}

\item{surv_model}{A linear model predicting survival probability. This can be
a model of class \code{glm} or \code{glmer}, and requires a predicted
binomial variable under a logit link. Ignored if \code{modelsuite} is
provided. This model must have been developed in a modeling exercise testing
only the impacts of occasion \emph{t}.}

\item{fec_model}{A linear model predicting fecundity. This can be a model of
class \code{glm}, \code{glmer}, \code{glmmTMB}, \code{zeroinfl},
\code{vglm}, \code{lm}, or \code{lmer}. Ignored if \code{modelsuite} is
provided. This model must have been developed in a modeling exercise testing
only the impacts of occasion \emph{t}.}

\item{paramnames}{A data frame with three columns, the first describing all
terms used in linear modeling, the second (must be called \code{mainparams})
giving the general model terms that will be used in matrix creation, and the
third showing the equivalent terms used in modeling (must be named
\code{modelparams}). Function \code{\link{create_pm}()} can be used to
create a skeleton \code{paramnames} object, which can then be edited. Only
required if \code{modelsuite} is not supplied.}

\item{start_age}{The age from which to start the matrix. Defaults to
\code{NA}, in which case age \code{1} is used if \code{prebreeding = TRUE},
and age \code{0} is used if \code{prebreeding = FALSE}.}

\item{last_age}{The final age to use in the matrix. Defaults to \code{NA}, in
which case the highest age in the dataset is used.}

\item{fecage_min}{The minimum age at which reproduction is possible. Defaults
to \code{NA}, which is interpreted to mean that fecundity should be assessed
starting in the minimum age observed in the dataset.}

\item{fecage_max}{The maximum age at which reproduction is possible. Defaults
to \code{NA}, which is interpreted to mean that fecundity should be assessed
until the final observed age.}

\item{continue}{A logical value designating whether to allow continued
survival of individuals past the final age noted in the stageframe, using the 
demographic characteristics of the final age. Defaults to \code{TRUE}.}

\item{inda}{Can be a single value to use for individual covariate \code{a}
in all matrices, or a vector of such values corresponding to each occasion in
the dataset. Defaults to \code{NULL}.}

\item{indb}{Can be a single value to use for individual covariate \code{b}
in all matrices, or a vector of such values corresponding to each occasion in
the dataset. Defaults to \code{NULL}.}

\item{indc}{Can be a single value to use for individual covariate \code{c}
in all matrices, or a vector of such values corresponding to each occasion in
the dataset. Defaults to \code{NULL}.}

\item{surv_dev}{A numeric value to be added to the y-intercept in the linear
model for survival probability. Defaults to \code{0}.}

\item{fec_dev}{A numeric value to be added to the y-intercept in the linear
model for fecundity. Defaults to \code{0}.}

\item{density}{A numeric value indicating density value to use to propagate
matrices. Only needed if density is an explanatory term used in linear
models. Defaults to \code{NA}.}

\item{repmod}{A scalar multiplier of fecundity. Defaults to \code{1}.}

\item{random.inda}{A logical value denoting whether to treat individual
covariate \code{a} as a random, categorical variable. Otherwise is treated as
a fixed, numeric variable. Defaults to \code{FALSE}.}

\item{random.indb}{A logical value denoting whether to treat individual
covariate \code{b} as a random, categorical variable. Otherwise is treated as
a fixed, numeric variable. Defaults to \code{FALSE}.}

\item{random.indc}{A logical value denoting whether to treat individual
covariate \code{c} as a random, categorical variable. Otherwise is treated as
a fixed, numeric variable. Defaults to \code{FALSE}.}

\item{negfec}{A logical value denoting whether fecundity values estimated to
be negative should be reset to \code{0}. Defaults to \code{FALSE}.}

\item{exp_tol}{A numeric value used to indicate a maximum value to set
exponents to in the core kernel to prevent numerical overflow. Defaults to
\code{700}.}

\item{theta_tol}{A numeric value used to indicate a maximum value to theta as
used in the negative binomial probability density kernel. Defaults to
\code{100000000}, but can be reset to other values during error checking.}
}
\value{
If all inputs are properly formatted, then this function will return
an object of class \code{lefkoMat}, which is a list that holds the matrix
projection model and all of its metadata. Its structure has the following
elements:

\item{A}{A list of full projection matrices in order of sorted patches and
occasions. All matrices output in R's \code{matrix} class.}
\item{U}{A list of survival transition matrices sorted as in \code{A}. All 
matrices output in R's \code{matrix} class.}
\item{F}{A list of fecundity matrices sorted as in \code{A}. All matrices 
output in R's \code{matrix} class.}
\item{hstages}{Set to \code{NA} for Leslie MPMs.}
\item{agestages}{Set to \code{NA} for Leslie MPMs.}
\item{ahstages}{A data frame detailing the characteristics of associated
ages, in the form of a modified stageframe including reproduction status.}
\item{labels}{A data frame giving the patch and year of each matrix in order.
In \code{fleslie()}, only one population may be analyzed at once.}
\item{matrixqc}{A short vector describing the number of non-zero elements
in \code{U} and \code{F} matrices, and the number of annual matrices.}
\item{modelqc}{This is the \code{qc} portion of the modelsuite input.}
}
\description{
Function \code{fleslie()} returns age-based (Leslie) MPMs corresponding to
the patches and occasions given, including the associated component
transition and fecundity matrices, data frames detailing the characteristics
of the exact ages corresponding to rows and columns in estimated matrices,
and a data frame characterizing the patch and occasion combinations
corresponding to these matrices.
}
\section{Notes}{

Unlike \code{\link{rlefko2}()}, \code{\link{rlefko3}()},
\code{\link{arlefko2}()}, and \code{\link{rleslie}()}, this function does not
currently distinguish populations.

This function will yield incorrect estimates if the models utilized
incorporate state in occasion \emph{t}-1, or any size or reproductive status
terms.

Users may at times wish to estimate MPMs using a dataset incorporating
multiple patches or subpopulations, but without discriminating between those
patches or subpopulations. Should the aim of analysis be a general MPM that
does not distinguish these patches or subpopulations, the
\code{modelsearch()} run should not include patch terms.

Input options including multiple variable names must be entered in the order
of variables in occasion \emph{t}+1 and \emph{t}. Rearranging the order will
lead to erroneous calculations, and may lead to fatal errors.

Care should be taken to match the random status of year and patch to the
states of those variables within the modelsuite. If they do not match, then
they will be treated as zeroes in vital rate estimation.

Individual covariates are treated as categorical only if they are set as
random terms. Fixed categorical individual covariates are currently not
allowed. However, such terms may be supplied if the \code{modelsuite} option
is set to a \code{vrm_input} object. In that case, the user should also set
the logical random switch for the individual covariate to be used to 
\code{TRUE} (e.g., \code{random.inda = TRUE}).
}

\examples{
\donttest{
data(lathyrus)

lathvert_base <- verticalize3(lathyrus, noyears = 4, firstyear = 1988,
  patchidcol = "SUBPLOT", individcol = "GENET", blocksize = 9,
  sizeacol = "Volume88", repstracol = "FCODE88", fecacol = "Intactseed88",
  deadacol = "Dead1988", censorcol = "Missing1988", censorkeep = NA,
  censor = TRUE, NRasRep = TRUE, NOasObs = TRUE)

lathvert_base$feca3 <- round(lathvert_base$feca3)
lathvert_base$feca2 <- round(lathvert_base$feca2)
lathvert_base$feca1 <- round(lathvert_base$feca1)

lathvert_age <- subset(lathvert_base, firstseen > 1988)

lathmodels2_age <- modelsearch(lathvert_age, historical = FALSE,
  approach = "mixed", suite = "cons", bestfit = "AICc&k", age = "obsage",
  vitalrates = c("surv", "fec"), fecdist = "poisson", indiv = "individ",
  year = "year2", show.model.tables = TRUE, quiet = TRUE)

lathmat2fleslie <- fleslie(year = "all", data = lathvert_age,
  modelsuite = lathmodels2_age, fecage_min = 1)

summary(lathmat2fleslie)
}
}
\seealso{
\code{\link{flefko3}()}

\code{\link{flefko2}()}

\code{\link{aflefko2}()}

\code{\link{arlefko2}()}

\code{\link{rlefko3}()}

\code{\link{rlefko2}()}

\code{\link{rleslie}()}
}
