% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eva.R
\name{eva}
\alias{eva}
\alias{alogLik.gevrFit}
\alias{alogLik.gpdFit}
\title{Loglikelihood adjustment for eva fits}
\usage{
\method{alogLik}{gevrFit}(x, cluster = NULL, use_vcov = TRUE, ...)

\method{alogLik}{gpdFit}(x, cluster = NULL, use_vcov = TRUE, ...)
}
\arguments{
\item{x}{A fitted model object with certain associated S3 methods.
See \strong{Details}.}

\item{cluster}{A vector or factor indicating from which cluster the
respective loglikelihood contributions from \code{loglik} originate.
This must have the same length as the vector returned by the
\code{logLikVec} method for an object like \code{x}.
If \code{cluster} is not supplied (i.e. is \code{NULL}) then it is
assumed that each observation forms its own cluster.
See \strong{Details}.}

\item{use_vcov}{A logical scalar.  Should we use the \code{vcov} S3 method
for \code{x} (if this exists) to estimate the Hessian of the independence
loglikelihood to be passed as the argument \code{H} to
\code{\link[chandwich]{adjust_loglik}}?
Otherwise, \code{H} is estimated inside
\code{\link[chandwich]{adjust_loglik}} using
\code{\link[stats:optim]{optimHess}}.}

\item{...}{Further arguments to be passed to the functions in the
sandwich package \code{\link[sandwich]{meat}} (if \code{cluster = NULL}),
or \code{\link[sandwich:vcovCL]{meatCL}} (if \code{cluster} is not
\code{NULL}).}
}
\value{
An object inheriting from class \code{"chandwich"}.  See
  \code{\link[chandwich]{adjust_loglik}}.
  \code{class(x)} is a vector of length 5. The first 3 components are
  \code{c("lax", "chandwich", "eva")}.
  The 4th component depends on which model was fitted.
  \code{"rlarg"} if \code{\link[eva]{gevrFit}} was used;
  \code{"gpd"} if \code{\link[eva]{gpdFit}} was used.
  The 5th component is
  \code{"stat"} if there are no covariates in the mode and
  \code{"nonstat"} otherwise.
}
\description{
S3 \code{alogLik} method to perform loglikelihood adjustment for fitted
extreme value model objects returned from the functions
\code{\link[eva]{gevrFit}} and \code{\link[eva]{gpdFit}} in the eva package.
}
\details{
See \code{\link{alogLik}} for details.

  In the stationary case (no covariates) the function
  \code{\link[eva]{gevrFit}} and \code{\link[eva]{gpdFit}} in the eva
  package offer standard errors based on the expected information or on the
  observed information, via the argument \code{information}.  In contrast,
  \code{alogLik()} always bases calculations on the observed information
  matrix. Therefore, unadjusted standard errors resulting from
  \code{alogLik()} may be different the corresponding standard errors
  from  \code{\link[eva]{gevrFit}} or \code{\link[eva]{gpdFit}}.

  For \code{\link[eva]{gevrFit}} only GEV fits (\code{gumbel = FALSE}) are
  supported.
}
\examples{
# We need the eva package
got_eva <- requireNamespace("eva", quietly = TRUE)

if (got_eva) {
  library(eva)
  # An example from the eva::gpdFit documentation
  set.seed(7)
  x <- eva::rgpd(2000, loc = 0, scale = 2, shape = 0.2)
  mle_fit <- eva::gpdFit(x, threshold = 4, method = "mle")
  adj_mle_fit <- alogLik(mle_fit)
  summary(adj_mle_fit)

  # Another example from the eva::gpdFit documentation
  # A linear trend in the scale parameter
  set.seed(7)
  n <- 300
  x2 <- eva::rgpd(n, loc = 0, scale = 1 + 1:n / 200, shape = 0)
  covs <- as.data.frame(seq(1, n, 1))
  names(covs) <- c("Trend1")
  result1 <- eva::gpdFit(x2, threshold = 0, scalevars = covs,
                         scaleform = ~ Trend1)
  adj_result1 <- alogLik(result1)
  summary(adj_result1)

  # An example from the eva::gevrFit documentation
  set.seed(7)
  x1 <- eva::rgevr(500, 1, loc = 0.5, scale = 1, shape = 0.3)
  result1 <- eva::gevrFit(x1, method = "mle")
  adj_result1 <- alogLik(result1)
  summary(adj_result1)

  # Another example from the eva::gevrFit documentation
  # A linear trend in the location and scale parameter
  n <- 100
  r <- 10
  x2 <- eva::rgevr(n, r, loc = 100 + 1:n / 50,  scale = 1 + 1:n / 300,
                   shape = 0)
  covs <- as.data.frame(seq(1, n, 1))
  names(covs) <- c("Trend1")
  # Create some unrelated covariates
  covs$Trend2 <- rnorm(n)
  covs$Trend3 <- 30 * runif(n)
  result2 <- eva::gevrFit(data = x2, method = "mle", locvars = covs,
                          locform = ~ Trend1 + Trend2*Trend3,
                          scalevars = covs, scaleform = ~ Trend1)
  adj_result2 <- alogLik(result2)
  summary(adj_result2)
}
}
\references{
Chandler, R. E. and Bate, S. (2007). Inference for clustered
  data using the independence loglikelihood. \emph{Biometrika},
  \strong{94}(1), 167-183. \doi{10.1093/biomet/asm015}

Zeileis (2006) Object-Oriented Computation and Sandwich
  Estimators.  \emph{Journal of Statistical Software}, \strong{16}, 1-16.
  \doi{10.18637/jss.v016.i09}
}
\seealso{
\code{\link{alogLik}}: loglikelihood adjustment for model fits.
}
