
###############################################################################
# Exact MISE for normal mixtures
###############################################################################


###############################################################################
# Omega matrices (for exact MISE for normal mixtures)
#
# Parameters 
# mus - means
# Sigmas - variances
# k - number of mixture components
# a - subscript of Omega matrix
# H - bandwidth matrix
#
# Returns 
# Omega matrix
###############################################################################

omega <- function(mus, Sigmas, k, a, H)
{
  # the (i,j) element of Omega matrix is dmvnorm(0, mu_i - mu_j,
  # a*H + Sigma_i + Sigma_j)
  
  if (k == 1)
    omega.mat <- dmvnorm(x=mus, mean=mus, sigma=a*H + 2*Sigmas)
  else
  {   
    if (is.matrix(mus)) d <- ncol(mus)
    else d <- length(mus)
    omega.mat <- matrix(0, nr=k, nc=k)
    for (i in 1:k)
    {
      Sigmai <- Sigmas[((i-1)*d+1):(i*d),]
      mui <- mus[i,]
      for (j in 1:k)
      {
        Sigmaj <- Sigmas[((j-1)*d+1):(j*d),]
        muj <- mus[j,]    
        omega.mat[i,j] <- dmvnorm(x=mui, mean=muj, sigma=a*H + Sigmai + Sigmaj)
      }
    }
  }
  
  return(omega.mat)
}


###############################################################################
# Lambda matrices (for exact AMISE for normal mixtures)
#
# Parameters 
# mus - means
# Sigmas - variances
# k - number of mixture components
# r - derivative (r1, r2)
#
# Returns 
# Lambda matrix
###############################################################################

lambda <- function(mus, Sigmas, k, r)
{
  # the (i,j) element of Lambda matrix is d^r/ dx^r  dmvnorm(0, mu_i - mu_j,
  # a*H + Sigma_i + Sigma_j)
  
  if (k == 1)
    lambda.mat <- dmvnorm.deriv.2d(r=r, x=rep(0, length(mus)), Sigma=2*Sigmas)
  else
  {   
    if (is.matrix(mus)) d <- ncol(mus)
    else d <- length(mus)
    lambda.mat <- matrix(0, nr=k, nc=k)
    for (i in 1:k)
    {
      Sigmai <- Sigmas[((i-1)*d+1) : (i*d),]
      mui <- mus[i,]
      for (j in 1:k)
      {
        Sigmaj <- Sigmas[((j-1)*d+1) : (j*d),]
        muj <- mus[j,]    
        lambda.mat[i,j] <- dmvnorm.deriv.2d(r=r, x=mui-muj,Sigma=Sigmai+Sigmaj)
      }
    }
  }
  
  return(lambda.mat)
}



##############################################################################
# Exact MISE for normal mixtures
#
# Parameters
# mus - means
# Sigmas - variances
# Props - vector of proportions of each mixture component 
# H - bandwidth matrix
# samp - sample size
#
# Returns
# Exact MISE for normal mixtures
###############################################################################

mise.mixt <- function(H, mus, Sigmas, props, samp)
{  
  if (is.vector(mus)) d <- length(mus)
  else d <- ncol(mus)
  k <- length(props)

  ## formula is found in Wand & Jones (1993)
  if (k == 1) 
  {
    mise <- 1/(samp * (4*pi)^(d/2) * sqrt(det(H))) + 
      (1-1/samp)*omega(mus, Sigmas, 1, 2, H) -
        2*omega(mus, Sigmas, 1, 1, H) +
        omega(mus, Sigmas, 1, 0, H)
  }
  else
  {
    mise <- 1/(samp * (4*pi)^(d/2) * sqrt(det(H))) +
      props %*% ((1-1/samp)*omega(mus, Sigmas, k, 2, H) - 
                 2*omega(mus, Sigmas, k, 1, H) + 
                 omega(mus, Sigmas, k, 0, H)) %*% props
  }
  return(drop(mise)) 
}


###############################################################################
# Exact AMISE for bivariate normal mixtures 
#
# Parameters
# mus - means
# Sigmas - variances
# props - mixing proportions 
# H - bandwidth matrix
# samp - sample size
#
# Returns   
# Exact AMISE for normal mixtures
###############################################################################

amise.mixt <- function(H, mus, Sigmas, props, samp)
{  
  d <- 2
  k <- length(props)
  
  h1 <- sqrt(H[1,1])
  h2 <- sqrt(H[2,2])
  h12 <- H[1,2]

  ## formula is found in Wand & Jones (1993)
  if (k == 1) 
  {
    amise <- 1/(samp * (4*pi)^(d/2) * sqrt(det(H))) + 
      1/4 * (lambda(mus, Sigmas, 1, r=c(4, 0))*H[1,1]^2 +
             2*lambda(mus, Sigmas, 1, r=c(2, 2))*H[1,1]*H[2,2] + 
             lambda(mus, Sigmas, 1, r=c(0, 4))*H[2,2]^2) 
  }
  else
  {
    amise <- 1/(samp * (4*pi)^(d/2) * sqrt(det(H))) +
      1/4 * props %*% (lambda(mus, Sigmas, k, r=c(4,0))*H[1,1]^2 +
                       2*lambda(mus, Sigmas, k, r=c(2, 2))*H[1,1]*H[2,2] + 
                       lambda(mus, Sigmas, k, r=c(0,4))*H[2,2]^2) %*% props
  }
  
  return(drop(amise)) 
}


###############################################################################
# Finds the bandwidth matrix that minimises the MISE for normal mixtures
#
# Parameters
# mus - means
# Sigmas - variances
# props - vector of proportions of each mixture component 
# Hstart - initial bandwidth matrix
# samp - sample size
# full - 1 minimise over full bandwidth matrices
#      - 0 minimise over diagonal bandwidth matrices
# 
# Returns
# H_MISE
###############################################################################
       
Hmise.mixt <- function(mus, Sigmas, props, samp, Hstart)
{   
  if (is.vector(mus)) d <- length(mus)
  else d <- ncol(mus) 
  seed <- 8326

  # use normal reference estimate as initial condition
  set.seed(seed)
  x <- rmvnorm.mixt(1000, mus, Sigmas, props)
  if (missing(Hstart))
    Hstart <- matrix.sqrt((4/(samp*(d + 2)))^(2/(d + 4)) * var(x))
    
  Hstart <- vech(Hstart)

  # input vech(H) into mise.mixt.temp because optim can only optimise
  # over vectors and not matrices
  mise.mixt.temp <- function(vechH)
  {  
    H <- invvech(vechH) %*% invvech(vechH)
    # using H <- invvech(vechH) %*% invvech(vechH) ensures that H
    # is positive definite
    
    return(mise.mixt(H=H, mus=mus, Sigmas=Sigmas, props=props, samp=samp))
  }

  result <- optim(Hstart, mise.mixt.temp, method = "BFGS")
  Hmise <- invvech(result$par) %*% invvech(result$par) 
  
  return(Hmise)
}   



###############################################################################
# Finds the bandwidth matrix that minimises the AMISE for normal mixtures
#
# Parameters
# mus - means
# Sigmas - variances
# props - vector of proportions of each mixture component 
# Hstart - initial bandwidth matrix
# samp - sample size
# 
# Returns
# Bandwidth matrix that minimises AMISE
###############################################################################
       
Hamise.mixt <- function(mus, Sigmas, props, samp, Hstart)
{   
  if (is.vector(mus)) d <- length(mus)
  else d <- ncol(mus) 
  seed <- 8326
  
  # use normal reference estimate as initial condition
  if (missing(Hstart)) 
  {
    set.seed(seed)
    x <- rmvnorm.mixt(1000, mus, Sigmas, props)
    Hstart <- matrix.sqrt((4/ (samp*(d + 2)))^(2/(d + 4)) * var(x))
  }
  
  # input vech(H) into mise.mixt.temp because optim can only optimise
  # over vectors and not matrices    
  Hstart <- vech(Hstart)
  amise.mixt.temp <- function(vechH)
  {
    H <- invvech(vechH) %*% invvech(vechH)
    # ensures that H is positive definite
      
    return(amise.mixt(H=H, mus=mus, Sigmas=Sigmas, props=props, samp=samp))
  }
    
  result <- optim(Hstart, amise.mixt.temp, method="BFGS")
  Hamise <- invvech(result$par) %*% invvech(result$par) 
      
  return(Hamise)
}   
  

###############################################################################
# ISE for normal mixtures (fixed KDE)
# 
# Parameters
# x - data values
# H - bandwidth matrix
# mus - matrix of means (each row is a vector of means from each component
#       density)
# Sigmas - matrix of covariance matrices (every d rows is a covariance matrix 
#          from each component density) 
# props - mixing proportions
# lower - vector of lower end points of rectangle
# upper - vector of upper end points of rectangle
# gridsize - vector of number of grid points
# stepsize - vector of step sizes
# Returns
# ISE 
###############################################################################

ise.mixt <- function(x, H, mus, Sigmas, props, lower,
                     upper, gridsize=c(250,250), stepsize=NULL)
{  
  if (is.list(x))
    return (ise.mixt.pc(x, H, mus, Sigmas, props, lower, upper, gridsize,
                        stepsize))
  if (is.vector(x)) x <- matrix(x,nr=1)
  if (is.vector(mus)) mus <- matrix(mus, nr=length(props))
  
  d <- ncol(x)
  n <- nrow(x)
  M <- length(props)
  ise1 <- 0
  ise2 <- 0
  ise3 <- 0

  # formula is found in thesis  
  if (d==2)
    ise1 <- dmvnorm.2d.sum(x=x, Sigma=2*H, inc=1)
  else if (d==4)
    ise1 <- dmvnorm.4d.sum(x=x, Sigma=2*H, inc=1)
  else if (d==6)
    ise1 <- dmvnorm.6d.sum(x=x, Sigma=2*H, inc=1)
  
  for (j in 1:M)
  {
    Sigmaj <- Sigmas[((j-1)*d+1) : (j*d),]
    ise2 <- ise2 + sum(props[j]*dmvnorm(x=x, mean=mus[j,], sigma=H + Sigmaj))
    
    for (i in 1:M)
    {
      Sigmai <- Sigmas[((i-1)*d+1) : (i*d),]
      ise3 <- ise3 + sum(props[i] * props[j] *
                         dmvnorm(x=mus[i,], mean=mus[j,], sigma=Sigmai+Sigmaj))
    }
  }  

  return (ise1/n^2 - 2*ise2/n + ise3)
}



###############################################################################
# ISE for t mixtures (numerical computation)
# 
# Parameters
# x - data values
# H - bandwidth matrices
# mus - matrix of means 
# Sigmas - matrix of covariance matrices 
# props - mixing proportions
# dfs - degrees of freedom
# lower, upper - lower and upper limits for integration
#
# Returns
# ISE  
###############################################################################

iset.mixt <- function(x, H, mus, Sigmas, dfs, props, lower,
                      upper, gridsize=c(250,250), stepsize=NULL) 
{
  if (!(identical(all.equal(sum(props), 1), TRUE)))   
    stop("Proportions don't sum to one\n")
  else if (length(dfs) != length(props))
    stop("Length of df and mixing proportions vectors not equal")  

  d <- ncol(Sigmas)

  # pre-clustered KDE
  if (is.list(x))
  {
    x1 <- x
    x <- x$x
    n <-  nrow(x)
    ind.lab <- sort(unique(x1$ind))
    Hs <- numeric(0)
    for (i in 1:n)
    {
      clust <- which(x1$ind[i]==ind.lab)
      H1 <- H[((clust-1)*d+1):(clust*d),]
      Hs <- rbind(Hs, H1)
    }
  }
  # fixed KDE
  else
  {
    n <-  nrow(x)
    Hs <- numeric(0)
    for (i in 1:n)
      Hs <- rbind(Hs, H) 
  }
  
  if(!is.null(gridsize))
  {  
    xx <- seq(lower[1], upper[1], length=gridsize[1])
    yy <- seq(lower[2], upper[2], length=gridsize[2])
  }
  else if (!is.null(stepsize))
  {
    xx <- seq(lower[1], upper[1], by=stepsize[1])
    yy <- seq(lower[2], upper[2], by=stepsize[2]) 
  } 
    
  xxyy <- permute(list(xx, yy))
  fhat <- dmvnorm.mixt(x=xxyy, mus=x, Sigma=Hs, props=rep(1/n,n))
  mixt <- dmvt.mixt(x=xxyy, mu=mus, Sigma=Sigmas, props=props, dfs=dfs)
  stepsize <- c(xx[1]-xx[2], yy[1]-yy[2])
  
  ise <- sum((fhat-mixt)^2*stepsize[1]*stepsize[2])

  return(ise)
}
