% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kdensity.R
\name{kdensity}
\alias{kdensity}
\title{Parametrically guided kernel density estimation}
\usage{
kdensity(
  x,
  bw = NULL,
  adjust = 1,
  kernel = NULL,
  start = NULL,
  support = NULL,
  na.rm = FALSE,
  normalized = TRUE,
  tolerance = 0.01
)
}
\arguments{
\item{x}{Numeric vector containing the data.}

\item{bw}{A bandwidth function. Can be either a string, a custom-made
function, or a double. The supported bandwidth functions are documented
in \code{\link[=bandwidths]{bandwidths()}}.}

\item{adjust}{An adjustment constant, so that \code{h = adjust*bw*sd}, where \code{sd}
varies with the chosen kernel.}

\item{kernel}{The kernel function. Can be chosen from the list of built-in
kernels or be custom-made. See \code{\link[=kernels]{kernels()}} for details.}

\item{start}{Parametric start. Can be chosen from the list of built-in
parametric starts or be custom-made. See \code{\link[=parametric_starts]{parametric_starts()}} for
details.}

\item{support}{The support of the data. Must be compatible with the supplied
\code{x} and the supplied \code{start} and \code{kernel}. Is used to find the
normalization constant, see \code{normalized}.}

\item{na.rm}{Logical; if \code{TRUE}, \code{NA}s will be removed from \code{x}.}

\item{normalized}{Logical; if \code{TRUE}, the density is normalized.}

\item{tolerance}{Numeric; the relative error to tolerate in normalization.}
}
\value{
\code{kdensity} returns an S3 function object of
\code{\link[base:class]{base::class()}} "kdensity". This is a callable function with the
following elements, accessible by '$':
\describe{
\item{\code{x}}{The data supplied in \code{x}.}
\item{\verb{bw_str, bw, adjust, h}}{The bandwidth function, the resulting
bandwidth, the \code{adjust} argument, and the adjusted
bandwidth.}
\item{\verb{kernel_str, kernel, start, start_str, support}}{Name of the kernel,
the kernel object, name of the parametric start, the start object,
and the support of the density.}
\item{\verb{data.name, n, range, has.na, na.rm, normalized}}{Name of the data, number of
observations, the range of the data, whether the data
\code{x} contained \code{NA} values, whether na.rm is \code{TRUE}
or not, and whether the density is normalized.}
\item{\code{call}}{The \code{call} to \code{kdensity}.}
\item{\code{estimates}}{Named numeric vector containing the parameter
estimates from the parametric start.}
\item{\code{logLik}}{The log-likelihood of the parametric starts. Is \code{NA}
for the uniform start.}

}
}
\description{
\code{kdensity} computes a parametrically guided kernel density estimate
for univariate data. It supports asymmetric kernels and parametric starts
through the \code{kernel} and \code{start} arguments.
}
\details{
The default values for \code{bw}, \code{kernel}, \code{start}, and
\code{support} are interdependent, and are chosen to make sense. E.g.,
the default value for \code{support} when \code{start = beta} is
\code{c(0, 1)}.

The \code{start} argument defaults to \code{uniform}, which corresponds
to ordinary kernel density estimation. The typical default value for
\code{kernel} is \code{gaussian}.

If \code{normalized} is \code{FALSE} and \code{start != "uniform"}, the resulting
density will not integrate to 1 in general.
}
\examples{
## Use gamma kernels to model positive data, the concentration of
## theophylline

concentration = Theoph$conc + 0.001
plot(kdensity(concentration, start = "gamma", kernel = "gamma", adjust = 1/3),
     ylim = c(0, 0.15), lwd = 2, main = "Concentration of theophylline")
lines(kdensity(concentration, start = "gamma", kernel = "gaussian"),
      lty = 2, col = "grey", lwd = 2)
lines(kdensity(concentration, start = "gaussian", kernel = "gaussian"),
      lty = 3, col = "blue", lwd = 2)
lines(kdensity(concentration, start = "gaussian", kernel = "gamma", adjust = 1/3),
      lty = 4, col = "red", lwd = 2)
rug(concentration)

## Using a density and and estimator from another package.

skew_hyperbolic = list(
  density   = SkewHyperbolic::dskewhyp,
  estimator = function(x) SkewHyperbolic::skewhypFit(x, printOut = FALSE)$param,
  support   = c(-Inf, Inf)
)

kde = kdensity(diff(LakeHuron), start = skew_hyperbolic)
plot(kde, lwd = 2, col = "blue",
     main = "Annual differences in water level (ft) of Lake Huron, 1875 - 1972")
lines(kde, plot_start = TRUE, lty = 2, lwd = 2) # Plots the skew hyperbolic density.
rug(diff(LakeHuron))

kde$estimates # Also: coef(kde)
# Displays the parameter estimates:
#        mu     delta      beta        nu
# -1.140713  3.301112  2.551657 26.462469

}
\references{
Hjort, Nils Lid, and Ingrid K. Glad. "Nonparametric density estimation with a parametric start." The Annals of Statistics (1995): 882-904.

Jones, M. C., and D. A. Henderson. "Miscellanea kernel-type density estimation on the unit interval." Biometrika 94.4 (2007): 977-984.

Chen, Song Xi. "Probability density function estimation using gamma kernels." Annals of the Institute of Statistical Mathematics 52.3 (2000): 471-480.

Silverman, Bernard W. Density estimation for statistics and data analysis. Vol. 26. CRC press, 1986.
}
\seealso{
The \code{stats} package function \code{\link[stats:density]{stats::density()}}.
}
