% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aftprobiths.R
\name{aftprobiths}
\alias{aftprobiths}
\title{Horseshoe shrinkage prior in integrated survival and binary regression}
\usage{
aftprobiths(ct, z, X, burn = 1000, nmc = 5000, thin = 1,
  alpha = 0.05, Xtest = NULL, cttest = NULL, ztest = NULL)
}
\arguments{
\item{ct}{survival response, a \eqn{n*2} matrix with first column as response and second column as right censored indicator,
1 is event time and 0 is right censored.}

\item{z}{binary response, a \eqn{n*1} vector with numeric values 0 or 1.}

\item{X}{Matrix of covariates, dimension \eqn{n*p}.}

\item{burn}{Number of burn-in MCMC samples. Default is 1000.}

\item{nmc}{Number of posterior draws to be saved. Default is 5000.}

\item{thin}{Thinning parameter of the chain. Default is 1 (no thinning).}

\item{alpha}{Level for the credible intervals. For example, alpha = 0.05 results in
95\% credible intervals.}

\item{Xtest}{test design matrix.}

\item{cttest}{test survival response.}

\item{ztest}{test binary response.}
}
\value{
\item{Beta.sHat}{Posterior mean of \eqn{\beta} for survival model, a \eqn{p} by 1 vector.}
\item{Beta.bHat}{Posterior mean of \eqn{\beta} for binary model, a \eqn{p} by 1 vector.}
\item{LeftCI.s}{The left bounds of the credible intervals for Beta.sHat.}
\item{RightCI.s}{The right bounds of the credible intervals for Beta.sHat.}
\item{LeftCI.b}{The left bounds of the credible intervals for Beta.bHat.}
\item{RightCI.b}{The right bounds of the credible intervals for Beta.bHat.}
\item{Beta.sMedian}{Posterior median of \eqn{beta} for survival model, a \eqn{p} by 1 vector.}
\item{Beta.bMedian}{Posterior median of \eqn{beta} for binary model, a \eqn{p} by 1 vector.}
\item{SigmaHat}{Posterior mean of variance covariance matrix.}
\item{LambdaHat}{Posterior mean of \eqn{\lambda}, a \eqn{p*1} vector.}
\item{TauHat}{Posterior mean of \eqn{\tau}, a \eqn{2*1} vector.}
\item{Beta.sSamples}{Posterior samples of \eqn{\beta} for survival model.}
\item{Beta.bSamples}{Posterior samples of \eqn{\beta} for binary model.}
\item{LambdaSamples}{Posterior samples of \eqn{\lambda}.}
\item{TauSamples}{Posterior samples of \eqn{\tau}.}
\item{SigmaSamples}{Posterior samples of variance covariance matrix.}
\item{DIC.s}{DIC for survival model.}
\item{DIC.b}{DIC for binary model.}
\item{SurvivalHat}{Predictive survival probability.}
\item{LogTimeHat}{Predictive log time.}
}
\description{
This function provides the implementation of integrated survival and binary high 
dimensiona regression utilizing Horseshoe prior on the paramters
}
\examples{
burnin <- 50
nmc    <- 150
thin <- 1
y.sd   <- 1  # standard deviation of the response

p <- 100  # number of predictors
ntrain <- 100  # training size
ntest  <- 50   # test size
n <- ntest + ntrain  # sample size
q <- 10   # number of true predictos

beta.t <- c(sample(x = c(1, -1), size = q, replace = TRUE), rep(0, p - q))  # randomly assign sign

Sigma <- matrix(0.9, nrow = p, ncol = p)
for(j in 1:p)
{
Sigma[j, j] <- 1
}

x <- mvtnorm::rmvnorm(n, mean = rep(0, p), sigma = Sigma)    # correlated design matrix

zmean <- x \%*\% beta.t
tmean <- x \%*\% beta.t
yCorr <- 0.5
yCov <- matrix(c(1, yCorr, yCorr, 1), nrow = 2)


y <- mvtnorm::rmvnorm(n, sigma = yCov)
t <- y[, 1] + tmean
z <- ifelse((y[, 2] + zmean) > 0, 1, 0)
X <- scale(as.matrix(x))  # standarization

z <- as.numeric(as.matrix(c(z)))
t <- as.numeric(as.matrix(c(t)))
T <- exp(t)   # AFT model
C <- rgamma(n, shape = 1.75, scale = 3)   # 42\% censoring time
time <- pmin(T, C)  # observed time is min of censored and true
status = time == T   # set to 1 if event is observed
ct <- as.matrix(cbind(time = time, status = status))  # censored time


# Training set
ztrain <- z[1:ntrain]
cttrain <- ct[1:ntrain, ]
Xtrain  <- X[1:ntrain, ]

# Test set
ztest <- z[(ntrain + 1):n]
cttest <- ct[(ntrain + 1):n, ]
Xtest  <- X[(ntrain + 1):n, ]

posterior.fit.joint <- aftprobiths(ct = cttrain, z = ztrain, X = Xtrain,
                                   burn = burnin, nmc = nmc, thin = thin,
                                   Xtest = Xtest, cttest = cttest, ztest = ztest)
                             
posterior.fit.joint$Beta.sHat

}
\references{
Maity, A. K., Carroll, R. J., and Mallick, B. K. (2019) 
            "Integration of Survival and Binary Data for Variable Selection and Prediction: 
            A Bayesian Approach", 
            Journal of the Royal Statistical Society: Series C (Applied Statistics).
}
