##' Get the Treatment Coefficients
##' 
##' Compute the overall coefficients every treatment term including the
##' interaction.
##' 
##' 
##' @param design.df a data frame containing the experimental design. Requires
##' every column be a \code{\link{factor}}.
##' @param trtTerm a vector of character containing the labels of the treatment
##' terms in the model generated by \code{\link{terms}}.
##' @return The numeric vector.
##' @author Kevin Chang
##' @examples
##' 
##' design1 <- local({ 
##'   Ani = as.factor(LETTERS[c(1,2,3,4,
##'                             5,6,7,8)])
##'   Trt = as.factor(letters[c(1,1,1,1,
##'                             2,2,2,2)])
##'   data.frame(Ani, Trt)
##' })
##' 
##' trt.str = "Trt"
##'   
##' fT = terms(as.formula(paste("~", trt.str, sep = "")), keep.order = TRUE)  #fixed terms
##' 
##' trtTerm = attr(fT,"term.labels")
##' effectsMatrix = attr(fT,"factor") 
##' 		
##' trt.Coef = getTrtCoef(design1, trtTerm)
##' 
##' 
##' @export getTrtCoef
getTrtCoef <- function(design.df, trtTerm) {
    
    if (length(trtTerm) == 1 && !any(grepl("[[:punct:]]", trtTerm))) {
        return(length(design.df[, trtTerm])/nlevels(design.df[, trtTerm]))
    } else if (any(grepl("[[:punct:]]", trtTerm))) {
        
        trtTermList <- lapply(strsplit(trtTerm, "[[:punct:]]+"), function(x) design.df[, 
            x])
        repList <- sapply(trtTermList, function(y) if (is.factor(y)) {
            mean(table(y))
        } else {
            mean(table(apply(y, 1, function(x) paste(x, collapse = "."))))
        })
        
        return(repList)
    } else {
        repList <- apply(design.df[, trtTerm], 2, function(x) mean(table(x)))
        return(repList)
    }
} 
