/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FLmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 

static size_t H5FL_reg_glb_mem_lim = 1 * 1024 * 1024;  
static size_t H5FL_reg_lst_mem_lim = 1 * 65536;        
static size_t H5FL_arr_glb_mem_lim = 4 * 1024 * 1024;  
static size_t H5FL_arr_lst_mem_lim = 4 * 65536;        
static size_t H5FL_blk_glb_mem_lim = 16 * 1024 * 1024; 
static size_t H5FL_blk_lst_mem_lim = 1024 * 1024; 
static size_t H5FL_fac_glb_mem_lim = 16 * 1024 * 1024; 
static size_t H5FL_fac_lst_mem_lim =
    1024 * 1024; 

typedef struct H5FL_reg_gc_node_t {
    H5FL_reg_head_t           *list; 
    struct H5FL_reg_gc_node_t *next; 
} H5FL_reg_gc_node_t;

typedef struct H5FL_reg_gc_list_t {
    size_t                     mem_freed; 
    struct H5FL_reg_gc_node_t *first; 
} H5FL_reg_gc_list_t;

static H5FL_reg_gc_list_t H5FL_reg_gc_head = {0, NULL};

typedef struct H5FL_gc_arr_node_t {
    H5FL_arr_head_t           *list; 
    struct H5FL_gc_arr_node_t *next; 
} H5FL_gc_arr_node_t;

typedef struct H5FL_gc_arr_list_t {
    size_t                     mem_freed; 
    struct H5FL_gc_arr_node_t *first; 
} H5FL_gc_arr_list_t;

static H5FL_gc_arr_list_t H5FL_arr_gc_head = {0, NULL};

typedef struct H5FL_blk_gc_node_t {
    H5FL_blk_head_t           *pq;   
    struct H5FL_blk_gc_node_t *next; 
} H5FL_blk_gc_node_t;

typedef struct H5FL_blk_gc_list_t {
    size_t                     mem_freed; 
    struct H5FL_blk_gc_node_t *first; 
} H5FL_blk_gc_list_t;

static H5FL_blk_gc_list_t H5FL_blk_gc_head = {0, NULL};

struct H5FL_fac_gc_node_t {
    H5FL_fac_head_t           *list; 
    struct H5FL_fac_gc_node_t *next; 
};

typedef struct H5FL_fac_gc_list_t {
    size_t                     mem_freed; 
    struct H5FL_fac_gc_node_t *first; 
} H5FL_fac_gc_list_t;

struct H5FL_fac_node_t {
    struct H5FL_fac_node_t *next; 
};

bool H5_PKG_INIT_VAR = false;

static H5FL_fac_gc_list_t H5FL_fac_gc_head = {0, NULL};

static void            *H5FL__malloc(size_t mem_size);
static herr_t           H5FL__reg_init(H5FL_reg_head_t *head);
static herr_t           H5FL__reg_gc(void);
static herr_t           H5FL__reg_gc_list(H5FL_reg_head_t *head);
static int              H5FL__reg_term(void);
static H5FL_blk_node_t *H5FL__blk_find_list(H5FL_blk_node_t **head, size_t size);
static H5FL_blk_node_t *H5FL__blk_create_list(H5FL_blk_node_t **head, size_t size);
static herr_t           H5FL__blk_init(H5FL_blk_head_t *head);
static herr_t           H5FL__blk_gc_list(H5FL_blk_head_t *head);
static herr_t           H5FL__blk_gc(void);
static int              H5FL__blk_term(void);
static herr_t           H5FL__arr_init(H5FL_arr_head_t *head);
static herr_t           H5FL__arr_gc_list(H5FL_arr_head_t *head);
static herr_t           H5FL__arr_gc(void);
static int              H5FL__arr_term(void);
static herr_t           H5FL__fac_gc_list(H5FL_fac_head_t *head);
static herr_t           H5FL__fac_gc(void);
static int              H5FL__fac_term_all(void);

H5FL_DEFINE(H5FL_blk_node_t);

H5FL_DEFINE_STATIC(H5FL_fac_gc_node_t);

H5FL_DEFINE(H5FL_fac_head_t);

int
H5FL_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        
        (void)H5FL_garbage_coll();

        
        n += H5FL__reg_term();
        n += H5FL__fac_term_all();
        n += H5FL__arr_term();
        n += H5FL__blk_term();

        
        if (0 == n)
            H5_PKG_INIT_VAR = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

static void *
H5FL__malloc(size_t mem_size)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ret_value = H5MM_malloc(mem_size))) {
        
        if (H5FL_garbage_coll() < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during allocation");

        
        if (NULL == (ret_value = H5MM_malloc(mem_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for chunk");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FL__reg_init(H5FL_reg_head_t *head)
{
    H5FL_reg_gc_node_t *new_node;            
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (new_node = (H5FL_reg_gc_node_t *)H5MM_malloc(sizeof(H5FL_reg_gc_node_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    new_node->list = head;

    
    new_node->next         = H5FL_reg_gc_head.first;
    H5FL_reg_gc_head.first = new_node;

    
    head->init = true;

    
    if (head->size < sizeof(H5FL_reg_node_t))
        head->size = sizeof(H5FL_reg_node_t);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_reg_free(H5FL_reg_head_t *head, void *obj)
{
    void *ret_value = NULL; 

    
    FUNC_ENTER_NOAPI_NOINIT

    
    assert(head);
    assert(obj);

#ifdef H5FL_DEBUG
    memset(obj, 255, head->size);
#endif 

    
    assert(head->init);

    
    ((H5FL_reg_node_t *)obj)->next = head->list;

    
    head->list = (H5FL_reg_node_t *)obj;

    
    head->onlist++;

    
    H5FL_reg_gc_head.mem_freed += head->size;

    
    
    if (head->onlist * head->size > H5FL_reg_lst_mem_lim)
        if (H5FL__reg_gc_list(head) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during free");

    
    if (H5FL_reg_gc_head.mem_freed > H5FL_reg_glb_mem_lim)
        if (H5FL__reg_gc() < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during free");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_reg_malloc(H5FL_reg_head_t *head)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(head);

    
    if (!head->init)
        if (H5FL__reg_init(head) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, NULL, "can't initialize 'regular' blocks");

    
    if (head->list != NULL) {
        
        ret_value = (void *)(head->list);

        
        head->list = head->list->next;

        
        head->onlist--;

        
        H5FL_reg_gc_head.mem_freed -= (head->size);
    } 
    
    else {
        if (NULL == (ret_value = H5FL__malloc(head->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

        
        head->allocated++;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_reg_calloc(H5FL_reg_head_t *head)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(head);

    
    if (NULL == (ret_value = H5FL_reg_malloc(head)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    memset(ret_value, 0, head->size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FL__reg_gc_list(H5FL_reg_head_t *head)
{
    H5FL_reg_node_t *free_list; 

    FUNC_ENTER_PACKAGE_NOERR

    
    free_list = head->list;
    while (free_list != NULL) {
        H5FL_reg_node_t *tmp; 

        
        tmp = free_list->next;

        
        H5MM_free(free_list);

        
        free_list = tmp;
    } 

    
    head->allocated -= head->onlist;

    
    H5FL_reg_gc_head.mem_freed -= (head->onlist * head->size);

    
    head->list   = NULL;
    head->onlist = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FL__reg_gc(void)
{
    H5FL_reg_gc_node_t *gc_node;             
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    gc_node = H5FL_reg_gc_head.first;
    while (gc_node != NULL) {
        
        if (H5FL__reg_gc_list(gc_node->list) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "garbage collection of list failed");

        
        gc_node = gc_node->next;
    } 

    
    assert(H5FL_reg_gc_head.mem_freed == 0);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FL__reg_term(void)
{
    H5FL_reg_gc_node_t *left; 

    FUNC_ENTER_PACKAGE_NOERR

    
    left = NULL;
    while (H5FL_reg_gc_head.first != NULL) {
        H5FL_reg_gc_node_t *tmp; 

        
        tmp = H5FL_reg_gc_head.first->next;

#ifdef H5FL_DEBUG
        Rprintf("%s: head->name = %s, head->allocated = %d\n", __func__, H5FL_reg_gc_head.first->list->name,
               (int)H5FL_reg_gc_head.first->list->allocated);
#endif 
        
        if (H5FL_reg_gc_head.first->list->allocated > 0) {
            
            H5FL_reg_gc_head.first->next = left;
            left                         = H5FL_reg_gc_head.first;
        } 
        
        else {
            
            H5FL_reg_gc_head.first->list->init = false;

            
            H5MM_xfree(H5FL_reg_gc_head.first);
        } 

        H5FL_reg_gc_head.first = tmp;
    } 

    
    H5FL_reg_gc_head.first = left;

    FUNC_LEAVE_NOAPI(H5FL_reg_gc_head.first != NULL ? 1 : 0)
} 

static H5FL_blk_node_t *
H5FL__blk_find_list(H5FL_blk_node_t **head, size_t size)
{
    H5FL_blk_node_t *temp = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    temp = *head;

    
    if (temp && temp->size != size) {
        temp = temp->next;

        while (temp != NULL) {
            
            if (temp->size == size) {
                
                if (temp->next == NULL) {
                    temp->prev->next = NULL;
                } 
                else {
                    temp->prev->next = temp->next;
                    temp->next->prev = temp->prev;
                } 

                
                temp->prev    = NULL;
                temp->next    = *head;
                (*head)->prev = temp;
                *head         = temp;

                
                break;
            } 

            temp = temp->next;
        } 
    }     

    FUNC_LEAVE_NOAPI(temp)
} 

static H5FL_blk_node_t *
H5FL__blk_create_list(H5FL_blk_node_t **head, size_t size)
{
    H5FL_blk_node_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ret_value = H5FL_CALLOC(H5FL_blk_node_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL, "memory allocation failed for chunk info");

    
    ret_value->size = size;

    
    if (NULL == *head)
        *head = ret_value;
    else {
        ret_value->next = *head;
        (*head)->prev   = ret_value;
        *head           = ret_value;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FL__blk_init(H5FL_blk_head_t *head)
{
    H5FL_blk_gc_node_t *new_node;            
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (new_node = (H5FL_blk_gc_node_t *)H5MM_malloc(sizeof(H5FL_blk_gc_node_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    new_node->pq = head;

    
    new_node->next         = H5FL_blk_gc_head.first;
    H5FL_blk_gc_head.first = new_node;

    
    head->init = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5FL_blk_free_block_avail(H5FL_blk_head_t *head, size_t size)
{
    H5FL_blk_node_t *free_list;        
    htri_t           ret_value = FAIL; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(head);

    
    
    if ((free_list = H5FL__blk_find_list(&(head->head), size)) != NULL && free_list->list != NULL)
        ret_value = true;
    else
        ret_value = false;

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_blk_malloc(H5FL_blk_head_t *head, size_t size)
{
    H5FL_blk_node_t *free_list;        
    H5FL_blk_list_t *temp;             
    void            *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(head);
    assert(size);

    
    if (!head->init)
        if (H5FL__blk_init(head) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, NULL, "can't initialize 'block' list");

    
    
    if (NULL != (free_list = H5FL__blk_find_list(&(head->head), size)) && NULL != free_list->list) {
        
        temp            = free_list->list;
        free_list->list = free_list->list->next;

        
        free_list->onlist--;
        head->onlist--;
        head->list_mem -= size;

        
        H5FL_blk_gc_head.mem_freed -= size;
    } 
    
    else {
        
        if (NULL == free_list)
            
            free_list = H5FL__blk_create_list(&(head->head), size);
        assert(free_list);

        
        if (NULL == (temp = (H5FL_blk_list_t *)H5FL__malloc(sizeof(H5FL_blk_list_t) + size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for chunk");

        
        free_list->allocated++;

        
        head->allocated++;
    } 

    
    temp->size = size;

    
    ret_value = ((char *)temp) + sizeof(H5FL_blk_list_t);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_blk_calloc(H5FL_blk_head_t *head, size_t size)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(head);
    assert(size);

    
    if (NULL == (ret_value = H5FL_blk_malloc(head, size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    memset(ret_value, 0, size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_blk_free(H5FL_blk_head_t *head, void *block)
{
    H5FL_blk_node_t *free_list;        
    H5FL_blk_list_t *temp;             
    size_t           free_size;        
    void            *ret_value = NULL; 

    
    FUNC_ENTER_NOAPI_NOINIT

    
    assert(head);
    assert(block);

    
    temp = (H5FL_blk_list_t *)((void *)((unsigned char *)block - sizeof(H5FL_blk_list_t)));

    
    free_size = temp->size;

#ifdef H5FL_DEBUG
    memset(temp, 255, free_size + sizeof(H5FL_blk_list_t));
#endif 

    
    if (NULL == (free_list = H5FL__blk_find_list(&(head->head), free_size)))
        
        free_list = H5FL__blk_create_list(&(head->head), free_size);
    if (NULL == free_list)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL, "couldn't create new list node");

    
    temp->next      = free_list->list; 
    free_list->list = temp;

    
    free_list->onlist++;
    head->onlist++;
    head->list_mem += free_size;

    
    H5FL_blk_gc_head.mem_freed += free_size;

    
    
    if (head->list_mem > H5FL_blk_lst_mem_lim)
        if (H5FL__blk_gc_list(head) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during free");

    
    if (H5FL_blk_gc_head.mem_freed > H5FL_blk_glb_mem_lim)
        if (H5FL__blk_gc() < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during free");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_blk_realloc(H5FL_blk_head_t *head, void *block, size_t new_size)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(head);
    assert(new_size);

    
    if (block != NULL) {
        H5FL_blk_list_t *temp; 

        
        temp = (H5FL_blk_list_t *)((void *)((unsigned char *)block - sizeof(H5FL_blk_list_t)));

        
        if (new_size != temp->size) {
            size_t blk_size; 

            if (NULL == (ret_value = H5FL_blk_malloc(head, new_size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for block");
            blk_size = MIN(new_size, temp->size);
            H5MM_memcpy(ret_value, block, blk_size);
            H5FL_blk_free(head, block);
        } 
        else
            ret_value = block;
    } 
    
    else
        ret_value = H5FL_blk_malloc(head, new_size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FL__blk_gc_list(H5FL_blk_head_t *head)
{
    H5FL_blk_node_t *blk_head; 

    FUNC_ENTER_PACKAGE_NOERR

    
    blk_head = head->head;
    while (blk_head != NULL) {
        H5FL_blk_node_t *blk_next; 
        H5FL_blk_list_t *list;     

        
        assert((blk_head->onlist && blk_head->list) || (0 == blk_head->onlist && NULL == blk_head->list));

        
        list = blk_head->list;
        while (list != NULL) {
            H5FL_blk_list_t *next; 

            
            next = list->next;

            
            H5MM_free(list);

            
            list = next;
        } 

        
        blk_head->allocated -= blk_head->onlist;
        head->allocated -= blk_head->onlist;

        
        head->list_mem -= (blk_head->onlist * blk_head->size);

        
        H5FL_blk_gc_head.mem_freed -= (blk_head->onlist * blk_head->size);

        
        blk_head->list   = NULL;
        blk_head->onlist = 0;

        
        blk_next = blk_head->next;

        
        if (0 == blk_head->allocated) {
            
            if (head->head == blk_head)
                head->head = blk_head->next;
            if (blk_head->prev)
                blk_head->prev->next = blk_head->next;
            if (blk_head->next)
                blk_head->next->prev = blk_head->prev;

            
            H5FL_FREE(H5FL_blk_node_t, blk_head);
        } 

        
        blk_head = blk_next;
    } 

    
    head->onlist = 0;

    
    assert(0 == head->list_mem);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FL__blk_gc(void)
{
    H5FL_blk_gc_node_t *gc_node;             
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    gc_node = H5FL_blk_gc_head.first;
    while (gc_node != NULL) {
        
        if (H5FL__blk_gc_list(gc_node->pq) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "garbage collection of list failed");

        
        gc_node = gc_node->next;
    } 

    
    assert(H5FL_blk_gc_head.mem_freed == 0);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FL__blk_term(void)
{
    H5FL_blk_gc_node_t *left; 

    FUNC_ENTER_PACKAGE_NOERR

    
    left = NULL;
    while (H5FL_blk_gc_head.first != NULL) {
        H5FL_blk_gc_node_t *tmp; 

        tmp = H5FL_blk_gc_head.first->next;

#ifdef H5FL_DEBUG
        Rprintf("%s: head->name = %s, head->allocated = %d\n", __func__, H5FL_blk_gc_head.first->pq->name,
               (int)H5FL_blk_gc_head.first->pq->allocated);
#endif 

        
        if (H5FL_blk_gc_head.first->pq->allocated > 0) {
            
            H5FL_blk_gc_head.first->next = left;
            left                         = H5FL_blk_gc_head.first;
        } 
        
        else {
            
            H5FL_blk_gc_head.first->pq->init = false;

            
            H5MM_free(H5FL_blk_gc_head.first);
        } 

        H5FL_blk_gc_head.first = tmp;
    } 

    
    H5FL_blk_gc_head.first = left;

    FUNC_LEAVE_NOAPI(H5FL_blk_gc_head.first != NULL ? 1 : 0)
} 

static herr_t
H5FL__arr_init(H5FL_arr_head_t *head)
{
    H5FL_gc_arr_node_t *new_node;            
    size_t              u;                   
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (new_node = (H5FL_gc_arr_node_t *)H5MM_malloc(sizeof(H5FL_gc_arr_node_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    new_node->list = head;

    
    new_node->next         = H5FL_arr_gc_head.first;
    H5FL_arr_gc_head.first = new_node;

    
    if (NULL ==
        (head->list_arr = (H5FL_arr_node_t *)H5MM_calloc((size_t)head->maxelem * sizeof(H5FL_arr_node_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    for (u = 0; u < (size_t)head->maxelem; u++)
        head->list_arr[u].size = head->base_size + (head->elem_size * u);

    
    head->init = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_arr_free(H5FL_arr_head_t *head, void *obj)
{
    H5FL_arr_list_t *temp;             
    size_t           mem_size;         
    size_t           free_nelem;       
    void            *ret_value = NULL; 

    
    FUNC_ENTER_NOAPI_NOINIT

    
    if (!obj)
        HGOTO_DONE(NULL);

    
    assert(head);

    
    assert(head->init);

    
    temp = (H5FL_arr_list_t *)((void *)((unsigned char *)obj - sizeof(H5FL_arr_list_t)));

    
    free_nelem = temp->nelem;

    
    assert((int)free_nelem <= head->maxelem);

    
    temp->next = head->list_arr[free_nelem].list;

    
    head->list_arr[free_nelem].list = temp;

    
    mem_size = head->list_arr[free_nelem].size;

    
    head->list_arr[free_nelem].onlist++;
    head->list_mem += mem_size;

    
    H5FL_arr_gc_head.mem_freed += mem_size;

    
    
    if (head->list_mem > H5FL_arr_lst_mem_lim)
        if (H5FL__arr_gc_list(head) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during free");

    
    if (H5FL_arr_gc_head.mem_freed > H5FL_arr_glb_mem_lim)
        if (H5FL__arr_gc() < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during free");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_arr_malloc(H5FL_arr_head_t *head, size_t elem)
{
    H5FL_arr_list_t *new_obj;          
    size_t           mem_size;         
    void            *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(head);
    assert(elem);

    
    if (!head->init)
        if (H5FL__arr_init(head) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, NULL, "can't initialize 'array' blocks");

    
    assert(elem <= (unsigned)head->maxelem);

    
    mem_size = head->list_arr[elem].size;

    
    if (head->list_arr[elem].list != NULL) {
        
        new_obj = head->list_arr[elem].list;

        
        head->list_arr[elem].list = head->list_arr[elem].list->next;

        
        head->list_arr[elem].onlist--;
        head->list_mem -= mem_size;

        
        H5FL_arr_gc_head.mem_freed -= mem_size;

    } 
    
    else {
        if (NULL == (new_obj = (H5FL_arr_list_t *)H5FL__malloc(sizeof(H5FL_arr_list_t) + mem_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

        
        head->list_arr[elem].allocated++;

        
        head->allocated++;
    } 

    
    new_obj->nelem = elem;

    
    ret_value = ((char *)new_obj) + sizeof(H5FL_arr_list_t);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_arr_calloc(H5FL_arr_head_t *head, size_t elem)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(head);
    assert(elem);

    
    if (NULL == (ret_value = H5FL_arr_malloc(head, elem)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    memset(ret_value, 0, head->list_arr[elem].size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_arr_realloc(H5FL_arr_head_t *head, void *obj, size_t new_elem)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(head);
    assert(new_elem);

    
    if (obj == NULL)
        ret_value = H5FL_arr_malloc(head, new_elem);
    else {
        H5FL_arr_list_t *temp; 

        
        assert((int)new_elem <= head->maxelem);

        
        temp = (H5FL_arr_list_t *)((void *)((unsigned char *)obj - sizeof(H5FL_arr_list_t)));

        
        if (temp->nelem != new_elem) {
            size_t blk_size; 

            
            ret_value = H5FL_arr_malloc(head, new_elem);

            
            blk_size = head->list_arr[MIN(temp->nelem, new_elem)].size;
            H5MM_memcpy(ret_value, obj, blk_size);

            
            H5FL_arr_free(head, obj);
        } 
        else
            ret_value = obj;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FL__arr_gc_list(H5FL_arr_head_t *head)
{
    unsigned u; 

    FUNC_ENTER_PACKAGE_NOERR

    
    for (u = 0; u < (unsigned)head->maxelem; u++) {
        if (head->list_arr[u].onlist > 0) {
            H5FL_arr_list_t *arr_free_list; 

            
            arr_free_list = head->list_arr[u].list;
            while (arr_free_list != NULL) {
                H5FL_arr_list_t *tmp; 

                
                tmp = arr_free_list->next;

                
                H5MM_free(arr_free_list);

                
                arr_free_list = tmp;
            } 

            
            head->list_arr[u].allocated -= head->list_arr[u].onlist;
            head->allocated -= head->list_arr[u].onlist;

            
            head->list_mem -= (head->list_arr[u].onlist * head->list_arr[u].size);

            
            H5FL_arr_gc_head.mem_freed -= (head->list_arr[u].onlist * head->list_arr[u].size);

            
            head->list_arr[u].list   = NULL;
            head->list_arr[u].onlist = 0;
        } 
    }     

    
    assert(head->list_mem == 0);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FL__arr_gc(void)
{
    H5FL_gc_arr_node_t *gc_arr_node;         
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    gc_arr_node = H5FL_arr_gc_head.first;
    while (gc_arr_node != NULL) {
        
        if (H5FL__arr_gc_list(gc_arr_node->list) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "garbage collection of list failed");

        
        gc_arr_node = gc_arr_node->next;
    } 

    
    assert(H5FL_arr_gc_head.mem_freed == 0);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FL__arr_term(void)
{
    H5FL_gc_arr_node_t *left; 

    FUNC_ENTER_PACKAGE_NOERR

    
    left = NULL;
    while (H5FL_arr_gc_head.first != NULL) {
        H5FL_gc_arr_node_t *tmp; 

        tmp = H5FL_arr_gc_head.first->next;

        
#ifdef H5FL_DEBUG
        Rprintf("%s: head->name = %s, head->allocated = %d\n", __func__, H5FL_arr_gc_head.first->list->name,
               (int)H5FL_arr_gc_head.first->list->allocated);
#endif 
        if (H5FL_arr_gc_head.first->list->allocated > 0) {
            
            H5FL_arr_gc_head.first->next = left;
            left                         = H5FL_arr_gc_head.first;
        } 
        
        else {
            
            H5MM_xfree(H5FL_arr_gc_head.first->list->list_arr);

            
            H5FL_arr_gc_head.first->list->init = false;

            
            H5MM_free(H5FL_arr_gc_head.first);
        } 

        H5FL_arr_gc_head.first = tmp;
    } 

    
    H5FL_arr_gc_head.first = left;

    FUNC_LEAVE_NOAPI(H5FL_arr_gc_head.first != NULL ? 1 : 0)
} 

void *
H5FL_seq_free(H5FL_seq_head_t *head, void *obj)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(head);
    assert(obj);

    
    assert(head->queue.init);

    
    H5FL_blk_free(&(head->queue), obj);

    FUNC_LEAVE_NOAPI(NULL)
} 

void *
H5FL_seq_malloc(H5FL_seq_head_t *head, size_t elem)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(head);
    assert(elem);

    
    ret_value = H5FL_blk_malloc(&(head->queue), head->size * elem);

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_seq_calloc(H5FL_seq_head_t *head, size_t elem)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(head);
    assert(elem);

    
    ret_value = H5FL_blk_calloc(&(head->queue), head->size * elem);

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_seq_realloc(H5FL_seq_head_t *head, void *obj, size_t new_elem)
{
    void *ret_value = NULL; 

    FUNC_ENTER_NOAPI_NOERR

    
    assert(head);
    assert(new_elem);

    
    ret_value = H5FL_blk_realloc(&(head->queue), obj, head->size * new_elem);

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FL_fac_head_t *
H5FL_fac_init(size_t size)
{
    H5FL_fac_gc_node_t *new_node  = NULL; 
    H5FL_fac_head_t    *factory   = NULL; 
    H5FL_fac_head_t    *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(size > 0);

    
    if (NULL == (factory = (H5FL_fac_head_t *)H5FL_CALLOC(H5FL_fac_head_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for factory object");

    
    factory->size = size;

    
    if (NULL == (new_node = (H5FL_fac_gc_node_t *)H5FL_MALLOC(H5FL_fac_gc_node_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    new_node->list = factory;

    
    new_node->next         = H5FL_fac_gc_head.first;
    H5FL_fac_gc_head.first = new_node;
    if (new_node->next)
        new_node->next->list->prev_gc = new_node;
    

    
    if (factory->size < sizeof(H5FL_fac_node_t))
        factory->size = sizeof(H5FL_fac_node_t);

    
    factory->init = true;

    
    ret_value = factory;

done:
    if (!ret_value) {
        if (factory)
            factory = H5FL_FREE(H5FL_fac_head_t, factory);
        if (new_node)
            new_node = H5FL_FREE(H5FL_fac_gc_node_t, new_node);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_fac_free(H5FL_fac_head_t *head, void *obj)
{
    void *ret_value = NULL; 

    
    FUNC_ENTER_NOAPI_NOINIT

    
    assert(head);
    assert(obj);

#ifdef H5FL_DEBUG
    memset(obj, 255, head->size);
#endif 

    
    assert(head->init);

    
    ((H5FL_fac_node_t *)obj)->next = head->list;

    
    head->list = (H5FL_fac_node_t *)obj;

    
    head->onlist++;

    
    H5FL_fac_gc_head.mem_freed += head->size;

    
    
    if (head->onlist * head->size > H5FL_fac_lst_mem_lim)
        if (H5FL__fac_gc_list(head) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during free");

    
    if (H5FL_fac_gc_head.mem_freed > H5FL_fac_glb_mem_lim)
        if (H5FL__fac_gc() < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, NULL, "garbage collection failed during free");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_fac_malloc(H5FL_fac_head_t *head)
{
    void *ret_value = NULL; 

    
    FUNC_ENTER_NOAPI_NOINIT

    
    assert(head);
    assert(head->init);

    
    if (head->list != NULL) {
        
        ret_value = (void *)(head->list);

        
        head->list = head->list->next;

        
        head->onlist--;

        
        H5FL_fac_gc_head.mem_freed -= (head->size);
    } 
    
    else {
        if (NULL == (ret_value = H5FL__malloc(head->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

        
        head->allocated++;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5FL_fac_calloc(H5FL_fac_head_t *head)
{
    void *ret_value = NULL; 

    
    FUNC_ENTER_NOAPI_NOINIT

    
    assert(head);

    
    if (NULL == (ret_value = H5FL_fac_malloc(head)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    memset(ret_value, 0, head->size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FL__fac_gc_list(H5FL_fac_head_t *head)
{
    H5FL_fac_node_t *free_list; 

    FUNC_ENTER_PACKAGE_NOERR

    
    free_list = head->list;
    while (free_list != NULL) {
        H5FL_fac_node_t *tmp; 

        
        tmp = free_list->next;

        
        H5MM_free(free_list);

        
        free_list = tmp;
    } 

    
    head->allocated -= head->onlist;

    
    H5FL_fac_gc_head.mem_freed -= (head->onlist * head->size);

    
    head->list   = NULL;
    head->onlist = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FL__fac_gc(void)
{
    H5FL_fac_gc_node_t *gc_node;             
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    gc_node = H5FL_fac_gc_head.first;
    while (gc_node != NULL) {
        
        if (H5FL__fac_gc_list(gc_node->list) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "garbage collection of list failed");

        
        gc_node = gc_node->next;
    } 

    
    assert(H5FL_fac_gc_head.mem_freed == 0);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FL_fac_term(H5FL_fac_head_t *factory)
{
    H5FL_fac_gc_node_t *tmp;                 
    herr_t              ret_value = SUCCEED; 

    
    FUNC_ENTER_NOAPI_NOINIT

    
    assert(factory);

    
    if (H5FL__fac_gc_list(factory) < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "garbage collection of factory failed");

    
    if (factory->allocated > 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "factory still has objects allocated");

    
    if (factory->prev_gc) {
        H5FL_fac_gc_node_t *last =
            factory->prev_gc; 

        assert(last->next->list == factory);
        tmp        = last->next->next;
        last->next = H5FL_FREE(H5FL_fac_gc_node_t, last->next);
        last->next = tmp;
        if (tmp)
            tmp->list->prev_gc = last;
    }
    else {
        assert(H5FL_fac_gc_head.first->list == factory);
        tmp                    = H5FL_fac_gc_head.first->next;
        H5FL_fac_gc_head.first = H5FL_FREE(H5FL_fac_gc_node_t, H5FL_fac_gc_head.first);
        H5FL_fac_gc_head.first = tmp;
        if (tmp)
            tmp->list->prev_gc = NULL;
    } 

    
    factory = H5FL_FREE(H5FL_fac_head_t, factory);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FL__fac_term_all(void)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    while (H5FL_fac_gc_head.first != NULL) {
        H5FL_fac_gc_node_t *tmp; 

        tmp = H5FL_fac_gc_head.first->next;

#ifdef H5FL_DEBUG
        Rprintf("%s: head->size = %d, head->allocated = %d\n", __func__,
               (int)H5FL_fac_gc_head.first->list->size, (int)H5FL_fac_gc_head.first->list->allocated);
#endif 

        
        assert(H5FL_fac_gc_head.first->list->allocated == 0);

        
        H5FL_fac_gc_head.first->list->init = false;

        
        H5FL_fac_gc_head.first = H5FL_FREE(H5FL_fac_gc_node_t, H5FL_fac_gc_head.first);

        H5FL_fac_gc_head.first = tmp;
    } 

    FUNC_LEAVE_NOAPI(0)
} 

herr_t
H5FL_garbage_coll(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5FL__arr_gc() < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "can't garbage collect array objects");

    
    if (H5FL__blk_gc() < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "can't garbage collect block objects");

    
    if (H5FL__reg_gc() < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "can't garbage collect regular objects");

    
    if (H5FL__fac_gc() < 0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGC, FAIL, "can't garbage collect factory objects");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FL_set_free_list_limits(int reg_global_lim, int reg_list_lim, int arr_global_lim, int arr_list_lim,
                          int blk_global_lim, int blk_list_lim, int fac_global_lim, int fac_list_lim)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOERR

    
    
    H5FL_reg_glb_mem_lim = (reg_global_lim == -1 ? UINT_MAX : (size_t)reg_global_lim);
    
    H5FL_reg_lst_mem_lim = (reg_list_lim == -1 ? UINT_MAX : (size_t)reg_list_lim);
    
    H5FL_arr_glb_mem_lim = (arr_global_lim == -1 ? UINT_MAX : (size_t)arr_global_lim);
    
    H5FL_arr_lst_mem_lim = (arr_list_lim == -1 ? UINT_MAX : (size_t)arr_list_lim);
    
    H5FL_blk_glb_mem_lim = (blk_global_lim == -1 ? UINT_MAX : (size_t)blk_global_lim);
    
    H5FL_blk_lst_mem_lim = (blk_list_lim == -1 ? UINT_MAX : (size_t)blk_list_lim);
    
    H5FL_fac_glb_mem_lim = (fac_global_lim == -1 ? UINT_MAX : (size_t)fac_global_lim);
    
    H5FL_fac_lst_mem_lim = (fac_list_lim == -1 ? UINT_MAX : (size_t)fac_list_lim);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FL_get_free_list_sizes(size_t *reg_size, size_t *arr_size, size_t *blk_size, size_t *fac_size)
{
    FUNC_ENTER_NOAPI_NOERR

    
    if (reg_size) {
        H5FL_reg_gc_node_t *gc_node; 

        
        *reg_size = 0;
        gc_node   = H5FL_reg_gc_head.first;
        while (gc_node != NULL) {
            H5FL_reg_head_t *reg_list = gc_node->list; 

            
            assert(reg_list->init);

            
            *reg_size += (reg_list->size * reg_list->allocated);

            
            gc_node = gc_node->next;
        } 
    }     

    
    if (arr_size) {
        H5FL_gc_arr_node_t *gc_arr_node; 

        
        *arr_size   = 0;
        gc_arr_node = H5FL_arr_gc_head.first;
        while (gc_arr_node != NULL) {
            H5FL_arr_head_t *head = gc_arr_node->list; 

            
            assert(head->init);

            
            if (head->allocated > 0) {
                unsigned u;

                
                for (u = 0; u < (unsigned)head->maxelem; u++)
                    
                    *arr_size += head->list_arr[u].allocated * head->list_arr[u].size;
            } 

            
            gc_arr_node = gc_arr_node->next;
        } 
    }     

    
    if (blk_size) {
        H5FL_blk_gc_node_t *gc_blk_node; 

        
        gc_blk_node = H5FL_blk_gc_head.first;
        *blk_size   = 0;
        while (gc_blk_node != NULL) {
            H5FL_blk_node_t *blk_head; 

            
            blk_head = gc_blk_node->pq->head;
            while (blk_head != NULL) {
                
                *blk_size += (blk_head->allocated * blk_head->size);

                
                blk_head = blk_head->next;
            } 

            
            gc_blk_node = gc_blk_node->next;
        } 
    }     

    
    if (fac_size) {
        H5FL_fac_gc_node_t *gc_fac_node; 

        
        gc_fac_node = H5FL_fac_gc_head.first;
        *fac_size   = 0;
        while (gc_fac_node != NULL) {
            H5FL_fac_head_t *fac_head = gc_fac_node->list; 

            
            *fac_size += (fac_head->allocated * fac_head->size);

            
            gc_fac_node = gc_fac_node->next;
        } 
    }     

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
