% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_wfs.R
\name{get_wfs}
\alias{get_wfs}
\title{Download WFS layer}
\usage{
get_wfs(shape = NULL,
        apikey = NULL,
        layer_name = NULL,
        filename = NULL,
        spatial_filter = "bbox",
        ecql_filter = NULL,
        overwrite = FALSE,
        interactive = FALSE)
}
\arguments{
\item{shape}{Object of class \code{sf}. Needs to be located in
France.}

\item{apikey}{API key from \code{get_apikeys()} or directly
from \href{https://geoservices.ign.fr/services-web-experts}{IGN website}}

\item{layer_name}{Name of the layer from \code{get_layers_metadata(apikey, "wfs")}
or directly from
\href{https://geoservices.ign.fr/services-web-experts}{IGN website}}

\item{filename}{Either a character string naming a file or a connection open
for writing. (ex : "test.shp" or "~/test.shp")}

\item{spatial_filter}{Character corresponding to a spatial predicate from ECQL language.
See detail and examples for more info.}

\item{ecql_filter}{Character corresponding to an ECQL query. See detail and examples
for more info.}

\item{overwrite}{If TRUE, file is overwrite}

\item{interactive}{if set to TRUE, no need to specify \code{apikey} and \code{layer_name}, you'll be ask.}
}
\value{
\code{get_wfs}return an object of class \code{sf}
}
\description{
Download a shapefile layer from IGN Web Feature Service (WFS).
To do that, it need a location giving by a shape, an apikey
and the name of layer. You can find those information from
\href{https://geoservices.ign.fr/services-web-experts}{IGN website}
}
\details{
\itemize{
\item \code{get_wfs} use ECQL language : a query language created by the OpenGeospatial Consortium.
It provide multiple spatial filter : "intersects", "disjoint", "contains", "within", "touches",
"crosses", "overlaps", "equals", "relate", "beyond", "dwithin". For "relate", "beyond",
"dwithin", argument can be provide using vector like :
spatial_filter = c("dwithin", distance, units). More info about ECQL language
\href{https://docs.geoserver.org/latest/en/user/filter/ecql_reference.html}{here}.
Be aware that "dwithin" is broken and it doesn't accept units properly. Only degrees can be used.
To avoid this, I recommend to use compute a buffer and use "within" instead od "dwithin".
\item ECQL query can be provided to \code{ecql_filter}. This allows direct query of the IGN's WFS
geoservers. If \code{shape} is set, then the \code{ecql_filter} comes in addition to the
\code{spatial_filter}. More info for writing ECQL \href{https://docs.geoserver.org/latest/en/user/tutorials/cql/cql_tutorial.html}{here}
}
}
\examples{
\dontrun{
library(sf)
library(tmap)

# shape from the best town in France
penmarch <- read_sf(system.file("extdata/penmarch.shp", package = "happign"))

# For quick testing, use interactive = TRUE
shape <- get_wfs(shape = penmarch,
                 interactive = TRUE)

# For specific use, choose apikey with get_apikey() and layer_name with get_layers_metadata()
## Getting borders of best town in France
apikey <- get_apikeys()[1]
metadata_table <- get_layers_metadata(apikey, "wfs")
layer_name <- metadata_table[32,1]

# Downloading borders
borders <- get_wfs(penmarch, apikey, layer_name)

# Plotting result
qtm(borders, fill = NULL, borders = "firebrick") # easy map

# Get forest_area of the best town in France
forest_area <- get_wfs(shape = borders,
                       apikey = "environnement",
                       layer_name = "LANDCOVER.FORESTINVENTORY.V1:resu_bdv1_shape")

qtm(forest_area, fill = "libelle")

# using ECQL filters to query IGN server

# find all commune's name starting by "plou".
# First you need the name of the attribute to filter
names(borders) # In our case "nom_m" is what we need

attribute_name <- names(get_wfs(penmarch,apikey, layer_name))
plou_borders <- get_wfs(shape = NULL, # When shape is NULL, all France is query
                        apikey = "administratif",
                        layer_name = "LIMITES_ADMINISTRATIVES_EXPRESS.LATEST:commune",
                        ecql_filter = "nom_m LIKE 'PLOU\%'")

# it's also possible to combine ecql_filter
plou_borders <- get_wfs(shape = NULL, # When shape is NULL, all France is query
                        apikey = "administratif",
                        layer_name = "LIMITES_ADMINISTRATIVES_EXPRESS.LATEST:commune",
                        ecql_filter = "nom_m LIKE 'PLOU\%' AND population < 2000")


}
}
\seealso{
\code{\link[=get_apikeys]{get_apikeys()}}, \code{\link[=get_layers_metadata]{get_layers_metadata()}}
}
