#' Get trip speed
#'
#' Returns the speed of each specified `trip_id`, based on the geometry created
#' from either the `shapes` or the `stop_times` file (or both).
#'
#' @param gtfs A GTFS object.
#' @param trip_id A character vector including the `trip_id`s to have their
#'   speeds calculated. If `NULL` (the default), the function calculates the
#'   speed of every `trip_id` in the GTFS.
#' @param file The file from which geometries should be generated, either
#'   `shapes` or `stop_times` (geometries are used to calculate the length of a
#'   trip). Defaults to `shapes`.
#' @param unit A string representing the unit in which the speeds are desired.
#'   Either `"km/h"` (the default) or `"m/s"`.
#'
#' @return A `data.table` containing the duration of each specified trip and the
#'   file from which geometries were generated.
#'
#' @section Details:
#' Please check [get_trip_geometry()] documentation to understand how geometry
#' generation differs depending on the chosen file.
#'
#' @seealso [get_trip_geometry()]
#'
#' @examples
#' data_path <- system.file("extdata/spo_gtfs.zip", package = "gtfstools")
#'
#' gtfs <- read_gtfs(data_path)
#'
#' # the examples below require the 'lwgeom' package to be installed
#' if (requireNamespace("lwgeom", quietly = TRUE)) {
#'
#'   trip_speed <- get_trip_speed(gtfs)
#'   head(trip_speed)
#'
#'   trip_ids <- c("CPTM L07-0", "2002-10-0")
#'   trip_speed <- get_trip_speed(gtfs, trip_ids)
#'   trip_speed
#'
#'   trip_speed <- get_trip_speed(
#'     gtfs,
#'     trip_ids,
#'     file = c("shapes", "stop_times")
#'   )
#'   trip_speed
#'
#'   trip_speed <- get_trip_speed(gtfs, trip_ids, unit = "m/s")
#'   trip_speed
#'
#' }
#'
#' @export
get_trip_speed <- function(gtfs,
                           trip_id = NULL,
                           file = "shapes",
                           unit = "km/h") {

  # checking if {lwgeom} is installed. {lwgeom} is a {sf} dependency required to
  # run sf::st_length()
  if (!requireNamespace("lwgeom", quietly = TRUE))
    stop(
      "The 'lwgeom' package is required to run this function. ",
      "Please install it first."
    )

  # input checking
  checkmate::assert_class(gtfs, "dt_gtfs")
  checkmate::assert_character(trip_id, null.ok = TRUE)
  checkmate::assert_names(file, subset.of = c("shapes", "stop_times"))
  checkmate::assert(
    checkmate::check_string(unit),
    checkmate::check_names(unit, subset.of = c("km/h", "m/s")),
    combine = "and"
  )

  # check if fields and files required to estimate trip duration exist (a bit of
  # an overlap, since these are also checked in a later get_trip_duration call,
  # but it prevents cases where the files required to generate geometries are
  # present but those required to estimate trip duration are not, which would
  # cause errors to be thrown very late)
  gtfsio::assert_field_class(
    gtfs,
    "stop_times",
    c("trip_id", "arrival_time", "departure_time"),
    rep("character", 3)
  )

  # generate desired geometries - checking for required files/fields is done
  # pretty early into get_trip_geometry code

  trips_geometries <- get_trip_geometry(gtfs, trip_id, file)

  # calculate the length of each geometry

  trips_length <- sf::st_length(trips_geometries)
  if (unit == "km/h") trips_length <- units::set_units(trips_length, "km")
  trips_length <- as.numeric(trips_length)

  # create data.table holding the length of each trip

  trips_speed <- data.table::data.table(
    trip_id     = trips_geometries$trip_id,
    origin_file = trips_geometries$origin_file,
    length      = trips_length
  )

  # calculate trips' duration
  # - a warning might be raised in get_trip_duration() if a trip in
  # existing_trips doesn't exist in stop_times table. if 'trip_id' is NULL, we
  # don't want to raise that warning, so we remove missing stop_times trips from
  # existing_trips

  existing_trips <- unique(trips_speed$trip_id)
  if (is.null(trip_id)) {
    stop_times_trips <- unique(gtfs$stop_times$trip_id)
    existing_trips <- existing_trips[existing_trips %chin% stop_times_trips]
  }

  duration_unit <- data.table::fifelse(unit == "km/h", "h", "s")

  trips_duration <- get_trip_duration(gtfs, existing_trips, duration_unit)

  # join trips_speed and trips_duration
  # a trip may be missing from trips_duration and not from trips_speed (when
  # stop_times doesn't contain a trip listed in trip_speed), but not the other
  # way around (because only trips listed in trips_speed have their durations
  # calculated). so we do a right join here, instead of a left join

  trips_speed <- trips_speed[trips_duration, on = "trip_id"]

  # calculate speed as length/duration

  trips_speed[, speed := length / duration]

  # remove length and duration columns

  trips_speed[, `:=`(length = NULL, duration = NULL)]

  return(trips_speed[])

}
