## ----setup, include = FALSE----------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width=6,
  fig.align = 'center'
)

## ----example-------------------------------------------------------------
library(igraph)   
library(ggraph)   
library(graphlayouts)

set.seed(666)
pa <- sample_pa(1000,1,1,directed = F)

ggraph(pa)+
  geom_edge_link(width=0.2,colour="grey")+
  geom_node_point(col="black",size=0.3)+
  theme_graph()

ggraph(pa,layout="stress")+
  geom_edge_link(width=0.2,colour="grey")+
  geom_node_point(col="black",size=0.3)+
  theme_graph()

## ----example_un----------------------------------------------------------
set.seed(666)
g <- disjoint_union(
  sample_pa(10,directed = F),
  sample_pa(20,directed = F),
  sample_pa(30,directed = F),
  sample_pa(40,directed = F),
  sample_pa(50,directed = F),
  sample_pa(60,directed = F),
  sample_pa(80,directed = F)
)

ggraph(g) +
  geom_edge_link() +
  geom_node_point() +
  theme_graph()

ggraph(g, layout="stress") +
  geom_edge_link() +
  geom_node_point() +
  theme_graph()

## ----hairball------------------------------------------------------------
set.seed(665)
#create network with a group structure
g <- sample_islands(9,40,0.4,15)
g <- simplify(g)
V(g)$grp <- as.character(rep(1:9,each=40))

ggraph(g,layout="stress")+
  geom_edge_link(colour=rgb(0,0,0,0.5),width=0.1)+
  geom_node_point(aes(col=grp))+
  scale_color_brewer(palette = "Set1")+
  theme_graph()+
  theme(legend.position = "none")


## ----backbone------------------------------------------------------------
bb <- layout_as_backbone(g,keep=0.4)
E(g)$col <- F
E(g)$col[bb$backbone] <- T

ggraph(g,layout="manual",node.positions=data.frame(x=bb$xy[,1],y=bb$xy[,2]))+
  geom_edge_link(aes(col=col),width=0.1)+
  geom_node_point(aes(col=grp))+
  scale_color_brewer(palette = "Set1")+
  scale_edge_color_manual(values=c(rgb(0,0,0,0.3),rgb(0,0,0,1)))+
  theme_graph()+
  theme(legend.position = "none")


## ----karate, include=FALSE-----------------------------------------------
el <- structure(c("Mr Hi", "Mr Hi", "Mr Hi", "Mr Hi", "Mr Hi", "Mr Hi", 
"Mr Hi", "Mr Hi", "Mr Hi", "Mr Hi", "Mr Hi", "Mr Hi", "Mr Hi", 
"Mr Hi", "Mr Hi", "Mr Hi", "Actor 2", "Actor 2", "Actor 2", "Actor 2", 
"Actor 2", "Actor 2", "Actor 2", "Actor 2", "Actor 3", "Actor 3", 
"Actor 3", "Actor 3", "Actor 3", "Actor 3", "Actor 3", "Actor 3", 
"Actor 4", "Actor 4", "Actor 4", "Actor 5", "Actor 5", "Actor 6", 
"Actor 6", "Actor 6", "Actor 7", "Actor 9", "Actor 9", "Actor 9", 
"Actor 10", "Actor 14", "Actor 15", "Actor 15", "Actor 16", "Actor 16", 
"Actor 19", "Actor 19", "Actor 20", "Actor 21", "Actor 21", "Actor 23", 
"Actor 23", "Actor 24", "Actor 24", "Actor 24", "Actor 24", "Actor 24", 
"Actor 25", "Actor 25", "Actor 25", "Actor 26", "Actor 27", "Actor 27", 
"Actor 28", "Actor 29", "Actor 29", "Actor 30", "Actor 30", "Actor 31", 
"Actor 31", "Actor 32", "Actor 32", "Actor 33", "Actor 2", "Actor 3", 
"Actor 4", "Actor 5", "Actor 6", "Actor 7", "Actor 8", "Actor 9", 
"Actor 11", "Actor 12", "Actor 13", "Actor 14", "Actor 18", "Actor 20", 
"Actor 22", "Actor 32", "Actor 3", "Actor 4", "Actor 8", "Actor 14", 
"Actor 18", "Actor 20", "Actor 22", "Actor 31", "Actor 4", "Actor 8", 
"Actor 9", "Actor 10", "Actor 14", "Actor 28", "Actor 29", "Actor 33", 
"Actor 8", "Actor 13", "Actor 14", "Actor 7", "Actor 11", "Actor 7", 
"Actor 11", "Actor 17", "Actor 17", "Actor 31", "Actor 33", "John A", 
"John A", "John A", "Actor 33", "John A", "Actor 33", "John A", 
"Actor 33", "John A", "John A", "Actor 33", "John A", "Actor 33", 
"John A", "Actor 26", "Actor 28", "Actor 30", "Actor 33", "John A", 
"Actor 26", "Actor 28", "Actor 32", "Actor 32", "Actor 30", "John A", 
"John A", "Actor 32", "John A", "Actor 33", "John A", "Actor 33", 
"John A", "Actor 33", "John A", "John A"), .Dim = c(78L, 2L))

karate <- graph_from_edgelist(el,directed = FALSE)
V(karate)$Faction <- c(1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 2, 2, 1, 1, 2, 1, 
2, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2)

## ----flex_focus----------------------------------------------------------
#library(igraphdata)
#data("karate")

p1 <- ggraph(karate,layout = "focus",v = 1) +
  draw_circle(use = "focus",max.circle = 3)+
  geom_edge_link(edge_color="black",edge_width=0.3)+
  geom_node_point(aes(fill=as.factor(Faction)),size=2,shape=21)+
  scale_fill_manual(values=c("#8B2323", "#EEAD0E"))+
  theme_graph()+
  theme(legend.position = "none")+
  coord_fixed()+
  labs(title= "Focus on Mr. Hi")

p2 <- ggraph(karate,layout = "focus",v = 34) +
  draw_circle(use = "focus",max.circle = 4)+
  geom_edge_link(edge_color="black",edge_width=0.3)+
  geom_node_point(aes(fill=as.factor(Faction)),size=2,shape=21)+
  scale_fill_manual(values=c("#8B2323", "#EEAD0E"))+
  theme_graph()+
  theme(legend.position = "none")+
  coord_fixed()+
  labs(title= "Focus on John A.")

p1
p2

## ----flex_cent-----------------------------------------------------------

bc <- betweenness(karate)
p1 <- ggraph(karate,layout = "centrality", cent = bc, tseq = seq(0,1,0.15)) +
  draw_circle(use = "cent") +
  annotate_circle(bc,format="",pos="bottom") +
  geom_edge_link(edge_color="black",edge_width=0.3)+
  geom_node_point(aes(fill=as.factor(Faction)),size=2,shape=21)+
  scale_fill_manual(values=c("#8B2323", "#EEAD0E"))+
  theme_graph()+
  theme(legend.position = "none")+
  coord_fixed()+
  labs(title="betweenness centrality")


cc <- closeness(karate)
p2 <- ggraph(karate,layout = "centrality", cent = cc, tseq = seq(0,1,0.2)) +
  draw_circle(use = "cent") +
  annotate_circle(cc,format="scientific",pos="bottom") +
  geom_edge_link(edge_color="black",edge_width=0.3)+
  geom_node_point(aes(fill=as.factor(Faction)),size=2,shape=21)+
  scale_fill_manual(values=c("#8B2323", "#EEAD0E"))+
  theme_graph()+
  theme(legend.position = "none")+
  coord_fixed()+
  labs(title="closeness centrality")

p1
p2

## ----eigen_layout--------------------------------------------------------
g <- sample_gnp(50,0.2)

ggraph(g,layout="eigen",type="adjacency",ev="largest")+
  geom_edge_link(n=2,edge_colour="grey66")+
  geom_node_point(shape=21,fill="grey25",size=5)+
  theme_graph()

ggraph(g,layout="eigen",type="adjacency",ev="smallest")+
  geom_edge_link(n=2,edge_colour="grey66")+
  geom_node_point(shape=21,fill="grey25",size=5)+
  theme_graph()


ggraph(g,layout="eigen",type="laplacian",ev="largest")+
  geom_edge_link(n=2,edge_colour="grey66")+
  geom_node_point(shape=21,fill="grey25",size=5)+
  theme_graph()

ggraph(g,layout="eigen",type="laplacian",ev="smallest")+
  geom_edge_link(n=2,edge_colour="grey66")+
  geom_node_point(shape=21,fill="grey25",size=5)+
  theme_graph()

