#' Equation, p-value, R^2, AIC or BIC from quantile regression
#'
#' \code{stat_quant_eq} fits a polynomial model by quantile regression and
#' generates several labels including the equation, p-value, coefficient of
#' determination (R^2), 'AIC' and 'BIC'.
#'
#' This statistic interprets the argument passed to \code{formula} differently
#' than \code{\link[ggplot2]{stat_quantile}} accepting \code{y} as well as
#' \code{x} as explanatory variable, matching \code{stat_poly_quant()}.
#'
#' When two variables are subject to mutual constrains, it is useful to consider
#' both of them as explanatory and interpret the relationship based on them. So,
#' from version 0.4.1 'ggpmisc' makes it possible to easily implement the
#' approach described by Cardoso (2019) under the name of "Double quantile
#' regression".
#'
#' @param mapping The aesthetic mapping, usually constructed with
#'   \code{\link[ggplot2]{aes}} or \code{\link[ggplot2]{aes_}}. Only needs to be
#'   set at the layer level if you are overriding the plot defaults.
#' @param data A layer specific dataset, only needed if you want to override
#'   the plot defaults.
#' @param geom The geometric object to use display the data
#' @param position The position adjustment to use for overlapping points on this
#'   layer
#' @param show.legend logical. Should this layer be included in the legends?
#'   \code{NA}, the default, includes if any aesthetics are mapped. \code{FALSE}
#'   never includes, and \code{TRUE} always includes.
#' @param inherit.aes If \code{FALSE}, overrides the default aesthetics, rather
#'   than combining with them. This is most useful for helper functions that
#'   define both data and aesthetics and shouldn't inherit behaviour from the
#'   default plot specification, e.g. \code{\link[ggplot2]{borders}}.
#' @param ... other arguments passed on to \code{\link[ggplot2]{layer}}. This
#'   can include aesthetics whose values you want to set, not map. See
#'   \code{\link[ggplot2]{layer}} for more details.
#' @param na.rm	a logical indicating whether NA values should be stripped before
#'   the computation proceeds.
#' @param formula a formula object. Using aesthetic names instead of
#'   original variable names.
#' @param quantiles numeric vector Values in 0..1 indicating the quantiles.
#' @param eq.with.lhs If \code{character} the string is pasted to the front of
#'   the equation label before parsing or a \code{logical} (see note).
#' @param eq.x.rhs \code{character} this string will be used as replacement for
#'   \code{"x"} in the model equation when generating the label before parsing
#'   it.
#' @param coef.digits,rho.digits integer Number of significant digits to use for
#'   the fitted coefficients and rho in labels.
#' @param coef.keep.zeros logical Keep or drop trailing zeros when formatting
#'   the fitted coefficients and F-value.
#' @param label.x,label.y \code{numeric} with range 0..1 "normalized parent
#'   coordinates" (npc units) or character if using \code{geom_text_npc()} or
#'   \code{geom_label_npc()}. If using \code{geom_text()} or \code{geom_label()}
#'   numeric in native data units. If too short they will be recycled.
#' @param label.x.npc,label.y.npc \code{numeric} with range 0..1 (npc units)
#'   DEPRECATED, use label.x and label.y instead; together with a geom
#'   using npcx and npcy aesthetics.
#' @param hstep,vstep numeric in npc units, the horizontal and vertical step
#'   used between labels for different groups.
#' @param output.type character One of \code{"expression"}, \code{"LaTeX"},
#'   \code{"text"}, \code{"markdown"} or \code{"numeric"}. In most cases,
#'   instead of using this statistics to obtain numeric values, it is better to
#'   use \code{stat_fit_tidy()}.
#' @param orientation character Either \code{"x"} or \code{"y"} controlling the
#'   default for \code{formula}.
#' @param parse logical Passed to the geom. If \code{TRUE}, the labels will be
#'   parsed into expressions and displayed as described in \code{?plotmath}.
#'   Default is \code{TRUE} if \code{output.type = "expression"} and
#'   \code{FALSE} otherwise.
#'
#' @note For backward compatibility a logical is accepted as argument for
#'   \code{eq.with.lhs}. If \code{TRUE}, the default is used, either
#'   \code{"x"} or \code{"y"}, depending on the argument passed to \code{formula}.
#'   However, \code{"x"} or \code{"y"} can be substituted by providing a
#'   suitable replacement character string through \code{eq.x.rhs}.
#'   Parameter \code{orientation} is redundant as it only affects the default
#'   for \code{formula} but is included for consistency with
#'   \code{ggplot2::stat_smooth()}.
#'
#' @details This stat can be used to automatically annotate a plot with R^2,
#'   adjusted R^2 or the fitted model equation. It supports only linear models
#'   fitted with function \code{lm()}. The R^2 and adjusted R^2 annotations can
#'   be used with any linear model formula. The fitted equation label is
#'   correctly generated for polynomials or quasi-polynomials through the
#'   origin. Model formulas can use \code{poly()} or be defined algebraically
#'   with terms of powers of increasing magnitude with no missing intermediate
#'   terms, except possibly for the intercept indicated by \code{"- 1"} or
#'   \code{"-1"} or \code{"+ 0"} in the formula. The validity of the
#'   \code{formula} is not checked in the current implementation, and for this
#'   reason the default aesthetics sets R^2 as label for the annotation.  This
#'   stat generates labels as R expressions by default but LaTeX (use TikZ
#'   device), markdown (use package 'ggtext') and plain text are also supported,
#'   as well as numeric values for user-generated text labels. The value of
#'   \code{parse} is set automatically based on \code{output-type}, but if you
#'   assemble labels that need parsing from \code{numeric} output, the default
#'   needs to be overriden. This stat only generates annotation labels, the
#'   predicted values/line need to be added to the plot as a separate layer
#'   using \code{\link{stat_quant_line}} or
#'   \code{\link[ggplot2]{stat_quantile}}, so to make sure that the same model
#'   formula is used in all steps it is best to save the formula as an object
#'   and supply this object as argument to the different statistics.
#'
#'   A ggplot statistic receives as data a data frame that is not the one passed
#'   as argument by the user, but instead a data frame with the variables mapped
#'   to aesthetics. stat_quant_eq() mimics how stat_smooth() works, except that
#'   only polynomials can be fitted. In other words, it respects the grammar of
#'   graphics. This helps ensure that the model is fitted to the same data as
#'   plotted in other layers.
#'
#' @references Written as an answer to question 65695409 by Mark Neal at
#'   Stackoverflow.
#'
#' @section Aesthetics: \code{stat_quant_eq} understands \code{x} and \code{y},
#'   to be referenced in the \code{formula} and \code{weight} passed as argument
#'   to parameter \code{weights} of \code{lm()}. All three must be mapped to
#'   \code{numeric} variables. In addition, the aesthetics undertood by the geom
#'   used (\code{"text"} by default) are understood and grouping respected.
#'
#' @section Computed variables:
#' If output.type different from \code{"numeric"} the returned tibble contains
#' columns below in addition to a modified version of the original \code{group}:
#' \describe{
#'   \item{x,npcx}{x position}
#'   \item{y,npcy}{y position}
#'   \item{eq.label}{equation for the fitted polynomial as a character string to be parsed}
#'   \item{rho.label}{\eqn{rho} of the fitted model as a character string to be parsed}
#'   \item{AIC.label}{AIC for the fitted model.}
#'   \item{n.label}{Number of observations used in the fit.}
#'   \item{rq.method}{character, method used.}
#'   \item{rho, n}{numeric values extracted or computed from fit object.}
#'   \item{hjust, vjust}{Set to "inward" to override the default of the "text" geom.}
#'   \item{quantile}{Numeric value of the quantile used for the fit}
#'   \item{quantile.f}{Factor with a level for each quantile}
#'   }
#'
#' If output.type is \code{"numeric"} the returned tibble contains columns
#'  in addition to a modified version of the original \code{group}:
#' \describe{
#'   \item{x,npcx}{x position}
#'   \item{y,npcy}{y position}
#'   \item{coef.ls}{list containing the "coefficients" matrix from the summary of the fit object}
#'   \item{rho, AIC, n}{numeric values extracted or computed from fit object}
#'   \item{rq.method}{character, method used.}
#'   \item{hjust, vjust}{Set to "inward" to override the default of the "text" geom.}
#'   \item{quantile}{Indicating the quantile  used for the fit}
#'   \item{quantile.f}{Factor with a level for each quantile}
#'   \item{b_0.constant}{TRUE is polynomial is forced through the origin}
#'   \item{b_i}{One or columns with the coefficient estimates}}
#'
#' To explore the computed values returned for a given input we suggest the use
#' of \code{\link[gginnards]{geom_debug}} as shown in the example below.
#'
#' @seealso This \code{stat_quant_eq} statistic can return ready formatted labels
#'   depending on the argument passed to \code{output.type}. This is possible
#'   because only polynomial models are supported. For other types of models,
#'   statistics \code{\link{stat_fit_glance}},  \code{\link{stat_fit_tidy}} and
#'   \code{\link{stat_fit_glance}} should be used instead and the code for
#'   construction of character strings from numeric values and their mapping to
#'   aesthetic \code{label} needs to be explicitly supplied in the call.
#'
#' @note Support for the \code{angle} aesthetic is not automatic and requires
#'   that the user passes as argument suitable numeric values to override the
#'   defaults for label positions.
#'
#' @family ggplot statistics for quantile regression
#'
#' @import quantreg
#'
#' @examples
#' # generate artificial data
#' set.seed(4321)
#' x <- 1:100
#' y <- (x + x^2 + x^3) + rnorm(length(x), mean = 0, sd = mean(x^3) / 4)
#' my.data <- data.frame(x = x, y = y,
#'                       group = c("A", "B"),
#'                       y2 = y * c(0.5,2),
#'                       w = sqrt(x))
#'
#' # using defaults
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line() +
#'   stat_quant_eq()
#'
#' # same formula as default
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = y ~ x) +
#'   stat_quant_eq(formula = y ~ x)
#'
#' # explicit formula "x explained by y"
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = x ~ y) +
#'   stat_quant_eq(formula = x ~ y)
#'
#' # using color
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(aes(color = after_stat(quantile.f))) +
#'   stat_quant_eq(aes(color = after_stat(quantile.f))) +
#'   labs(color = "Quantiles")
#'
#' # location and colour
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(aes(color = after_stat(quantile.f))) +
#'   stat_quant_eq(aes(color = after_stat(quantile.f)),
#'                 label.y = "bottom", label.x = "right") +
#'   labs(color = "Quantiles")
#'
#' # give a name to a formula
#' formula <- y ~ poly(x, 3, raw = TRUE)
#'
#' # no weights
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula) +
#'   stat_quant_eq(formula = formula)
#'
#' # angle
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula) +
#'   stat_quant_eq(formula = formula, angle = 90, hstep = 0.05, vstep = 0,
#'                 label.y = 0.98, hjust = 1)
#'
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula) +
#'   stat_quant_eq(formula = formula, angle = 90,
#'                 hstep = 0.05, vstep = 0, hjust = 0,
#'                 label.y = 0.25)
#'
#' # user set quantiles
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula, quantiles = 0.5) +
#'   stat_quant_eq(formula = formula, quantiles = 0.5)
#'
#' # grouping
#' ggplot(my.data, aes(x, y, color = group)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula) +
#'   stat_quant_eq(formula = formula)
#'
#' ggplot(my.data, aes(x, y, color = group)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula) +
#'   stat_quant_eq(formula = formula, angle = 90,
#'                 hstep = 0.05, vstep = 0, hjust = 0,
#'                 size = 3, label.y = 0.3)
#'
#' # labelling equations
#' ggplot(my.data, aes(x, y,  shape = group, linetype = group,
#'        grp.label = group)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula, color = "black") +
#'   stat_quant_eq(aes(label = paste(after_stat(grp.label), after_stat(eq.label), sep = "*\": \"*")),
#'                 formula = formula) +
#'   theme_classic()
#'
#' # setting non-default quantiles
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula,
#'                   quantiles = c(0.1, 0.5, 0.9)) +
#'   stat_quant_eq(formula = formula, parse = TRUE,
#'                 quantiles = c(0.1, 0.5, 0.9))
#'
#' # Location of equations
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula) +
#'   stat_quant_eq(formula = formula, label.y = "bottom", label.x = "right")
#'
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula) +
#'   stat_quant_eq(formula = formula, label.y = 0.03, label.x = 0.95, vstep = 0.04)
#'
#' # using weights
#' ggplot(my.data, aes(x, y, weight = w)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula) +
#'   stat_quant_eq(formula = formula)
#'
#' # no weights, quantile set to upper boundary
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(formula = formula, quantiles = 0.95) +
#'   stat_quant_eq(formula = formula, quantiles = 0.95)
#'
#' # user specified label
#' ggplot(my.data, aes(x, y, color = group, grp.label = group)) +
#'   geom_point() +
#'   stat_quant_line(method = "rq", formula = formula,
#'                 quantiles = c(0.05, 0.5, 0.95)) +
#'   stat_quant_eq(aes(label = paste(after_stat(grp.label), "*\": \"*",
#'                                    after_stat(eq.label), sep = "")),
#'                 quantiles = c(0.05, 0.5, 0.95),
#'                 formula = formula, size = 3)
#'
#' # geom = "text"
#' ggplot(my.data, aes(x, y)) +
#'   geom_point() +
#'   stat_quant_line(method = "rq", formula = formula, quantiles = 0.5) +
#'   stat_quant_eq(label.x = "left", label.y = "top",
#'                 formula = formula)
#'
#' # Inspecting the returned data using geom_debug()
#' \dontrun{
#' if (requireNamespace("gginnards", quietly = TRUE)) {
#'   library(gginnards)
#'
#' # This provides a quick way of finding out the names of the variables that
#' # are available for mapping to aesthetics.
#'
#'   ggplot(my.data, aes(x, y)) +
#'     geom_point() +
#'     stat_quant_eq(formula = formula, geom = "debug")
#'
#'   ggplot(my.data, aes(x, y)) +
#'     geom_point() +
#'     stat_quant_eq(aes(label = after_stat(eq.label)),
#'                   formula = formula, geom = "debug",
#'                   output.type = "markdown")
#'
#'   ggplot(my.data, aes(x, y)) +
#'     geom_point() +
#'     stat_quant_eq(formula = formula, geom = "debug", output.type = "text")
#'
#'   ggplot(my.data, aes(x, y)) +
#'     geom_point() +
#'     stat_quant_eq(formula = formula, geom = "debug", output.type = "numeric")
#'
#'   ggplot(my.data, aes(x, y)) +
#'     geom_point() +
#'     stat_quant_eq(formula = formula, quantiles = c(0.25, 0.5, 0.75),
#'                   geom = "debug", output.type = "text")
#'
#'   ggplot(my.data, aes(x, y)) +
#'     geom_point() +
#'     stat_quant_eq(formula = formula, quantiles = c(0.25, 0.5, 0.75),
#'                   geom = "debug", output.type = "numeric")
#' }
#' }
#'
#' @export
#'
stat_quant_eq <- function(mapping = NULL, data = NULL,
                         geom = "text_npc",
                         position = "identity",
                         ...,
                         formula = NULL,
                         quantiles = c(0.25, 0.5, 0.75),
                         eq.with.lhs = TRUE,
                         eq.x.rhs = NULL,
                         coef.digits = 3,
                         coef.keep.zeros = TRUE,
                         rho.digits = 2,
                         label.x = "left", label.y = "top",
                         label.x.npc = NULL, label.y.npc = NULL,
                         hstep = 0,
                         vstep = NULL,
                         output.type = "expression",
                         na.rm = FALSE,
                         orientation = NA,
                         parse = NULL,
                         show.legend = FALSE,
                         inherit.aes = TRUE) {
  # backwards compatibility
  if (!is.null(label.x.npc)) {
    stopifnot(grepl("_npc", geom))
    label.x <- label.x.npc
  }
  if (!is.null(label.y.npc)) {
    stopifnot(grepl("_npc", geom))
    label.y <- label.y.npc
  }
  if (is.null(parse)) {
    parse <- output.type == "expression"
  }
  ggplot2::layer(
    data = data,
    mapping = mapping,
    stat = StatQuantEq,
    geom = geom,
    position = position,
    show.legend = show.legend,
    inherit.aes = inherit.aes,
    params = list(formula = formula,
                  quantiles = quantiles,
                  eq.with.lhs = eq.with.lhs,
                  eq.x.rhs = eq.x.rhs,
                  coef.digits = coef.digits,
                  coef.keep.zeros = coef.keep.zeros,
                  rho.digits = rho.digits,
                  label.x = label.x,
                  label.y = label.y,
                  hstep = hstep,
                  vstep = ifelse(is.null(vstep),
                                 ifelse(grepl("label", geom),
                                        0.10,
                                        0.05),
                                 vstep),
                  npc.used = grepl("_npc", geom),
                  output.type = output.type,
                  na.rm = na.rm,
                  orientation = orientation,
                  parse = parse,
                  ...)
  )
}

# Defined here to avoid a note in check --as-cran as the import from 'polynom'
# is not seen when the function is defined in-line in the ggproto object.
#' @rdname ggpmisc-ggproto
#'
#' @format NULL
#' @usage NULL
#'
quant_eq_compute_group_fun <- function(data,
                                       scales,
                                       formula,
                                       quantiles,
                                       weight,
                                       eq.with.lhs,
                                       eq.x.rhs,
                                       coef.digits,
                                       coef.keep.zeros,
                                       rho.digits,
                                       label.x,
                                       label.y,
                                       hstep,
                                       vstep,
                                       npc.used,
                                       output.type,
                                       na.rm,
                                       orientation) {
  force(data)
  num.quantiles <- length(quantiles)

  # make sure quantiles are ordered
  quantiles <- sort(quantiles)

  # factor with nicely formatted labels
  quant.digits <- ifelse(min(quantiles) < 0.01 || max(quantiles) > 0.99, 3, 2)
  quant.levels <- sort(unique(quantiles), decreasing = TRUE)
  quant.labels <- sprintf("%#.*f", quant.digits, quant.levels)
  quantiles.f <- factor(quantiles,
                        levels = quant.levels,
                        labels = quant.labels)

  # we guess formula from orientation
  if (is.null(formula)) {
    if (is.na(orientation) || orientation == "x") {
      formula = y ~ x
    } else if (orientation == "y") {
      formula = x ~ y
    }
  }
  # we guess orientation from formula
  if (is.na(orientation)) {
    orientation <- unname(c(x = "y", y = "x")[as.character(formula)[2]])
  }

  output.type <- if (!length(output.type)) {
    "expression"
  } else {
    tolower(output.type)
  }
  stopifnot(output.type %in%
              c("expression", "text", "markdown", "numeric", "latex", "tex", "tikz"))

  if (is.null(data[["weight"]])) {
    data[["weight"]] <- 1
  }

  if (exists("grp.label", data)) {
    if (length(unique(data[["grp.label"]])) > 1L) {
    warning("Non-unique value in 'data$grp.label' for group.")
      grp.label <- ""
    } else {
      grp.label <- data[["grp.label"]][1]
    }
  } else {
    grp.label <- ""
  }

  group.idx <- abs(data[["group"]][1])
  if (length(label.x) >= group.idx) {
    label.x <- label.x[group.idx]
  } else if (length(label.x) > 0) {
    label.x <- label.x[1]
  }
  if (length(label.y) >= group.idx) {
    label.y <- label.y[group.idx]
  } else if (length(label.y) > 0) {
    label.y <- label.y[1]
  }

  if (orientation == "x") {
    if (length(unique(data[["x"]])) < 2) {
      warning("Not enough data to perform fit for group ",
              group.idx, "; computing mean instead.",
              call. = FALSE)
      formula = y ~ 1
    }
  } else if (orientation == "y") {
    if (length(unique(data[["y"]])) < 2) {
      warning("Not enough data to perform fit for group ",
              group.idx, "; computing mean instead.",
              call. = FALSE)
      formula = x ~ 1
    }
  }

  rq.args <- list(quote(formula),
                  tau = quantiles,
                  data = quote(data),
                  weights = quote(weight))

  # quantreg contains code with partial matching of names!
  # so we silence selectively only these warnings
  withCallingHandlers({
    mf <- do.call(quantreg::rq, rq.args)
    mf.summary <- summary(mf)
  }, warning = function(w) {
    if (startsWith(conditionMessage(w), "partial match of 'coef'") ||
        startsWith(conditionMessage(w), "partial argument match of 'contrasts'"))
      invokeRestart("muffleWarning")
  })

  if (class(mf.summary)[1L] == "summary.rq") {
    mf.summary <- list(mf.summary)
  }
  names(mf.summary) <- as.character(quantiles)

  AIC <- AIC(mf)
  n <- length(mf.summary[[1]][["residuals"]])
  rho <- mf[["rho"]]
  rq.method <- mf[["method"]]
  coefs.mt <- mf[["coefficients"]] # a matrix if length(quantiles) > 1
  # ensure that coefs.mt is consistent
  if (is.vector(coefs.mt)) {
    coefs.mt <- as.matrix(coefs.mt)
    colnames(coefs.mt) <- paste("tau=", mf[["tau"]], sep = "")
  }
  formula.rhs.chr <- as.character(formula)[3]
  forced.origin <- grepl("-[[:space:]]*1|+[[:space:]]*0", formula.rhs.chr)
  if (forced.origin) {
    coefs.mt <- rbind(rep(0, ncol(coefs.mt)), coefs.mt)
  }
  coefs.ls <- asplit(coefs.mt, 2)
  # located here so that names in coef.ls remain the same as in version 0.4.0
  rownames(coefs.mt) <-paste("b", (1:nrow(coefs.mt)) - 1, sep = "_")

  z <- tibble::tibble()
  if (output.type == "numeric") {
    z <- tibble::tibble(coef.ls = coefs.ls,
                        quantile = quantiles,
                        quantile.f = quantiles.f,
                        rq.method = rq.method,
                        AIC = AIC,
                        rho = rho,
                        n = n,
                        eq.label = "", # needed for default 'label' mapping
                        b_0.constant = forced.origin)
    z <- cbind(z, tibble::as_tibble(t(coefs.mt)))
  } else {
    # set defaults needed to assemble the equation as a character string
    if (is.null(eq.x.rhs)) {
      eq.x.rhs <- build_eq.x.rhs(output.type = output.type,
                                 orientation = orientation)
    }

    if (is.character(eq.with.lhs)) {
      lhs <- eq.with.lhs
      eq.with.lhs <- TRUE
    } else if (eq.with.lhs) {
      lhs <- build_lhs(output.type = output.type,
                       orientation = orientation)
    } else {
      lhs <- character(0)
    }

    # build labels
    stopifnot(coef.digits > 0)
    if (coef.digits < 3) {
      warning("'coef.digits < 3' Likely information loss!")
    }

    eq.char <- AIC.char <- rho.char <- character(num.quantiles)
    for (q in seq_along(quantiles)) {
      # build equation as a character string from the coefficient estimates
      eq.char[q] <- coefs2poly_eq(coefs = coefs.ls[[q]],
                                  coef.digits = coef.digits,
                                  coef.keep.zeros = coef.keep.zeros,
                                  eq.x.rhs = eq.x.rhs,
                                  lhs = lhs,
                                  output.type = output.type)

      if (output.type == "expression" && coef.keep.zeros) {
        AIC.char[q] <- sprintf("\"%.4g\"", AIC[q])
        rho.char[q] <- sprintf("\"%#.3g\"", rho[q])
      } else {
        AIC.char[q] <- sprintf("%.4g", AIC[q])
        rho.char[q] <- sprintf("%#.3g", rho[q])
      }
    }

    # build data frames to return
    if (output.type == "expression") {
      z <- tibble::tibble(eq.label = eq.char,
                          AIC.label = paste("AIC", AIC.char, sep = "~`=`~"),
                          rho.label = paste("rho", AIC.char, sep = "~`=`~"),
                          n.label = paste("italic(n)~`=`~", n, sep = ""),
                          grp.label = if (any(grp.label != ""))
                                         paste(grp.label,
                                            sprintf("italic(q)~`=`~%.2f", quantiles),
                                            sep = "*\", \"*")
                                      else
                                        sprintf("italic(q)~`=`~%.2f", quantiles),
                          rq.method = rq.method,
                          quantile = quantiles,
                          quantile.f = quantiles.f,
                          n = n)
    } else if (output.type %in% c("latex", "tex", "text", "tikz")) {
      z <- tibble::tibble(eq.label = eq.char,
                          AIC.label = paste("AIC", AIC.char, sep = " = "),
                          rho.label = paste("rho", AIC.char, sep = " = "),
                          n.label = paste("n = ", n, sep = ""),
                          grp.label = paste(grp.label,
                                            sprintf("q = %.2f", quantiles)),
                          rq.method = rq.method,
                          quantile = quantiles,
                          quantile.f = quantiles.f,
                          n = n)
    } else if (output.type == "markdown") {
      z <- tibble::tibble(eq.label = eq.char,
                          AIC.label = paste("AIC", AIC.char, sep = " = "),
                          rho.label = paste("rho", AIC.char, sep = " = "),
                          n.label = paste("_n_ = ", n, sep = ""),
                          grp.label = paste(grp.label,
                                            sprintf("q = %.2f", quantiles)),
                          rq.method = rq.method,
                          quantile = quantiles,
                          quantile.f = quantiles.f,
                          n = n)
    } else {
      warning("Unknown 'output.type' argument: ", output.type)
    }
  }

  # Compute label positions
  if (is.character(label.x)) {
    if (npc.used) {
      margin.npc <- 0.05
    } else {
      # margin set by scale
      margin.npc <- 0
    }
    label.x <-
      ggpp::compute_npcx(x = label.x, group = group.idx, h.step = hstep,
                         margin.npc = margin.npc, each.len = num.quantiles)
    if (!npc.used) {
      x.expanse <- abs(diff(range(data[["x"]])))
      x.min <- min(data[["x"]])
      label.x <- label.x * x.expanse + x.min
    }
  } else if (is.numeric(label.x) && length(label.x == 1L)) {
    if (!npc.used) {
      x.expanse <- abs(diff(range(data[["x"]])))
      x.min <- min(data[["x"]])
      x <- (label.x - x.min) / x.expanse
    } else {
      x <- label.x
    }
    group <- abs(group.idx)
    expanded.group <- integer()
    for (i in seq_along(group)) {
      temp <- seq(from = 1, by = 1, length.out = num.quantiles) +
        (group[i] - 1) * num.quantiles
      expanded.group <- c(expanded.group, temp)
    }
    if (any(expanded.group > 0L) && hstep != 0) {
      x <- x + (expanded.group - 1) * hstep * ifelse(x < 0.5, 1, -1)
    }
    x <- ifelse(x > 1, 1, x)
    x <- ifelse(x < 0, 0, x)
    if (!npc.used) {
      label.x <- x * x.expanse + x.min
    } else {
      label.x <- x
    }
  }

  if (is.character(label.y)) {
    rev.y.pos <- length(label.y) == 1L && label.y != "bottom"
    if (npc.used) {
      margin.npc <- 0.05
    } else {
      # margin set by scale
      margin.npc <- 0
    }
    label.y <-
      ggpp::compute_npcy(y = label.y, group = group.idx, v.step = vstep,
                         margin.npc = margin.npc, each.len = num.quantiles)
    if (!npc.used) {
      y.expanse <- abs(diff(range(data[["y"]])))
      y.min <- min(data[["y"]])
      label.y <- label.y * y.expanse + y.min
    }
  } else if (is.numeric(label.y) && length(label.y == 1L)) {
    rev.y.pos <- length(label.y) == 1L && label.y >= 0.5
    if (!npc.used) {
      y.expanse <- abs(diff(range(data[["y"]])))
      y.min <- min(data[["y"]])
      y <- (label.y - y.min) / y.expanse
    } else {
      y <- label.y
    }
    group <- abs(group.idx)
    expanded.group <- integer()
    for (i in seq_along(group)) {
      temp <- seq(from = 1, by = 1, length.out = num.quantiles) +
        (group[i] - 1) * num.quantiles
      expanded.group <- c(expanded.group, temp)
    }
    if (any(expanded.group > 0L) && vstep != 0) {
      y <- y + (expanded.group - 1) * vstep * ifelse(y < 0.5, 1, -1)
    }
    y <- ifelse(y > 1, 1, y)
    y <- ifelse(y < 0, 0, y)
    if (!npc.used) {
      label.y <- y * y.expanse + y.min
    } else {
      label.y <- y
    }
  }

  if (npc.used) {
    z[["npcx"]] <- label.x
    z[["x"]] <- NA_real_
    z[["npcy"]] <- if (rev.y.pos) rev(label.y) else label.y
    z[["y"]] <- NA_real_
  } else {
    z[["x"]] <- label.x
    z[["npcx"]] <- NA_real_
    z[["y"]] <- if (rev.y.pos) rev(label.y) else label.y
    z[["npcy"]] <- NA_real_
  }

  z
}

#' @rdname ggpmisc-ggproto
#' @format NULL
#' @usage NULL
#' @export
StatQuantEq <-
  ggplot2::ggproto("StatQuantEq", ggplot2::Stat,
                   extra_params = c("na.rm", "parse"),
                   compute_group = quant_eq_compute_group_fun,
                   default_aes =
                     ggplot2::aes(npcx = after_stat(npcx),
                                  npcy = after_stat(npcy),
                                  label = after_stat(eq.label),
                                  hjust = "inward",
                                  vjust = "inward"),
                   required_aes = c("x", "y")
  )

