% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convenience-functions.R
\name{prep_car_data}
\alias{prep_car_data}
\title{Prepare data for the CAR model}
\source{
Cliff A, Ord J (1981). Spatial Processes: Models and Applications. Pion.

Cressie N (2015 [1993]). Statistics for Spatial Data. Revised edition. John Wiley & Sons.

Cressie N, Perrin O, Thomas-Agnan C (2005). “Likelihood-based estimation for Gaussian MRFs.” Statistical Methodology, 2(1), 1–16.

Cressie N, Wikle CK (2011). Statistics for Spatio-Temporal Data. John Wiley & Sons.

Donegan, Connor (2021). Spatial conditional autoregressive models in Stan. \emph{OSF Preprints}. \doi{10.31219/osf.io/3ey65}.

Haining RP, Li G (2020). Modelling Spatial and Spatio-Temporal Data: A Bayesian Approach. CRC Press.
}
\usage{
prep_car_data(
  A,
  style = c("WCAR", "ACAR", "DCAR"),
  k = 1,
  gamma = 0,
  lambda = TRUE,
  cmat = TRUE,
  stan_fn = ifelse(style == "WCAR", "wcar_normal_lpdf", "car_normal_lpdf")
)
}
\arguments{
\item{A}{Binary adjacency matrix; for \code{style = DCAR}, provide a symmetric matrix of distances instead. The distance matrix should be sparse, meaning that most distances should be zero (usually obtained by setting some threshold distance beyond which all are zero).}

\item{style}{Specification for the connectivity matrix (C) and conditional variances (M); one of "WCAR", "ACAR", or "DCAR".}

\item{k}{For \code{style = DCAR}, distances will be raised to the -k power (d^-k).}

\item{gamma}{For \code{style = DCAR}, distances will be offset by \code{gamma} before raising to the \code{-k}th power.}

\item{lambda}{If TRUE, return eigenvalues required for calculating the log determinant of the precision matrix and for determining the range of permissible values of rho. These will also be printed with a message if lambda = TRUE.}

\item{cmat}{If \code{cmat = TRUE}, return the full matrix C (in sparse matrix format).}

\item{stan_fn}{Two computational methods are available for CAR models using \code{\link[geostan]{stan_car}}: \code{car\_normal\_lpdf} and \code{wcar\_normal\_lpdf}. For WCAR models, either method will work but \code{wcar\_normal\_lpdf} is faster. To force use \code{car\_normal\_lpdf} when \code{style = 'WCAR'}, provide \code{stan_fn = "car_normal_lpdf"}.}
}
\value{
A list containing all of the data elements required by the CAR model in \code{\link[geostan]{stan_car}}.
}
\description{
Prepare data for the CAR model
}
\details{
The CAR model is:

\if{html}{\out{<div class="sourceCode">}}\preformatted{  Normal(Mu, Sigma), Sigma = (I - rho * C)^-1 * M * tau^2,
}\if{html}{\out{</div>}}

where \code{I} is the identity matrix, \code{rho} is a spatial autocorrelation parameter, \code{C} is a connectivity matrix, and \code{M * tau^2} is a diagonal matrix with conditional variances on the diagonal. \code{tau^2} is a (scalar) scale parameter.

In the WCAR specification, \code{C} is the row-standardized version of \code{A}. This means that the non-zero elements of \code{A} will be converted to \code{1/N_i} where \code{N_i} is the number of neighbors for the \code{i}th site (obtained using \code{Matrix::rowSums(A)}. The conditional variances (on the diagonal of \code{M * tau^2}), are also proportional to \code{1/N_i}.

The ACAR specification is from Cressie, Perrin and Thomas-Agnon (2005); also see Cressie and Wikle (2011, p. 188) and Donegan (2021).

The DCAR specification is inverse distance-based, and requires the user provide a (sparse) distance matrix instead of a binary adjacency matrix. (For \code{A}, provide a symmetric matrix of distances, not inverse distances!) Internally, non-zero elements of \code{A} will be converted to: \verb{d_\{ij\} = (a_\{ij\} + gamma)^(-k)} (Cliff and Ord 1981, p. 144; Donegan 2021). Default values are \code{k=1} and \code{gamma=0}. Following Cressie (2015), these values will be scaled (divided) by their maximum value. For further details, see the DCAR_A specification in Donegan (2021).

For inverse-distance weighting schemes, see Cliff and Ord (1981); for distance-based CAR specifications, see Cressie (2015 [1993]), Haining and Li (2020), and Donegan (2021).

When using \code{\link[geostan]{stan_car}}, always use \code{cmat = TRUE} (the default).

Details on CAR model specifications can be found in Table 1 of Donegan (2021).
}
\examples{
data(georgia)

## use a binary adjacency matrix
A <- shape2mat(georgia, style = "B")

## WCAR specification
cp <- prep_car_data(A, "WCAR")
1 / range(cp$lambda)

## ACAR specification
cp <- prep_car_data(A, "ACAR")

## DCAR specification (inverse-distance based)
A <- shape2mat(georgia, "B")
D <- sf::st_distance(sf::st_centroid(georgia))
A <- D * A
cp <- prep_car_data(A, "DCAR", k = 1)
}
