% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stan_esf.R
\name{stan_esf}
\alias{stan_esf}
\title{Spatial filtering}
\source{
Chun, Y., D. A. Griffith, M. Lee and P. Sinha (2016). Eigenvector selection with stepwise regression techniques to construct eigenvector spatial filters. \emph{Journal of Geographical Systems}, 18(1), 67-85. \doi{10.1007/s10109-015-0225-3}.

Dray, S., P. Legendre & P. R. Peres-Neto (2006). Spatial modelling: a comprehensive framework for principal coordinate analysis of neighbour matrices (PCNM). \emph{Ecological Modeling}, 196(3-4), 483-493.

Donegan, C., Y. Chun and A. E. Hughes (2020). Bayesian estimation of spatial filters with Moran’s Eigenvectors and hierarchical shrinkage priors. \emph{Spatial Statistics}. \doi{10.1016/j.spasta.2020.100450} (open access: \doi{10.31219/osf.io/fah3z}).

Donegan, Connor and Chun, Yongwan and Griffith, Daniel A. (2021). Modeling community health with areal data: Bayesian inference with survey standard errors and spatial structure. \emph{Int. J. Env. Res. and Public Health} 18 (13): 6856. DOI: 10.3390/ijerph18136856 Data and code: \url{https://github.com/ConnorDonegan/survey-HBM}.

Donegan, Connor (2021). Spatial conditional autoregressive models in Stan. \emph{OSF Preprints}. \doi{10.31219/osf.io/3ey65}.

Griffith, Daniel A., and P. R. Peres-Neto (2006). Spatial modeling in ecology: the flexibility of eigenfunction spatial analyses. \emph{Ecology} 87(10), 2603-2613.

Griffith, D., and Y. Chun (2014). Spatial autocorrelation and spatial filtering, Handbook of Regional Science. Fischer, MM and Nijkamp, P. eds.

Griffith, D., Chun, Y. and Li, B. (2019). \emph{Spatial Regression Analysis Using Eigenvector Spatial Filtering}. Elsevier.

Piironen, J and A. Vehtari (2017). Sparsity information and regularization in the horseshoe and other shrinkage priors. In \emph{Electronic Journal of Statistics}, 11(2):5018-5051.
}
\usage{
stan_esf(
  formula,
  slx,
  re,
  data,
  C,
  EV = make_EV(C, nsa = nsa, threshold = threshold),
  nsa = FALSE,
  threshold = 0.25,
  family = gaussian(),
  prior = NULL,
  ME = NULL,
  centerx = FALSE,
  censor_point,
  prior_only = FALSE,
  chains = 4,
  iter = 2000,
  refresh = 500,
  keep_all = FALSE,
  pars = NULL,
  control = NULL,
  ...
)
}
\arguments{
\item{formula}{A model formula, following the R \code{\link[stats]{formula}} syntax. Binomial models are specified by setting the left hand side of the equation to a data frame of successes and failures, as in \code{cbind(successes, failures) ~ x}.}

\item{slx}{Formula to specify any spatially-lagged covariates. As in, \code{~ x1 + x2} (the intercept term will be removed internally). When setting priors for \code{beta}, remember to include priors for any SLX terms.}

\item{re}{To include a varying intercept (or "random effects") term, \code{alpha_re}, specify the grouping variable here using formula syntax, as in \code{~ ID}. Then, \code{alpha_re} is a vector of parameters added to the linear predictor of the model, and:\preformatted{alpha_re ~ N(0, alpha_tau)
alpha_tau ~ Student_t(d.f., location, scale).
}}

\item{data}{A \code{data.frame} or an object coercible to a data frame by \code{as.data.frame} containing the model data.}

\item{C}{Spatial connectivity matrix which will be used to calculate eigenvectors, if \code{EV} is not provided by the user. Typically, the binary connectivity matrix is best for calculating eigenvectors (i.e., using \code{C = shape2mat(shape, style = "B")}). This matrix will also be used to calculate residual spatial autocorrelation and any user specified \code{slx} terms; it will be row-standardized before calculating \code{slx} terms. See \code{\link[geostan]{shape2mat}}.}

\item{EV}{A matrix of eigenvectors from any (transformed) connectivity matrix, presumably spatial (see \code{\link[geostan]{make_EV}}). If \code{EV} is provided, still also provide a spatial weights matrix \code{C} for other purposes; \code{threshold} and \code{nsa} are ignored for user provided \code{EV}.}

\item{nsa}{Include eigenvectors representing negative spatial autocorrelation? Defaults to \code{nsa = FALSE}. This is ignored if \code{EV} is provided.}

\item{threshold}{Eigenvectors with standardized Moran coefficient values below this \code{threshold} value will be excluded from the candidate set of eigenvectors, \code{EV}. This defaults to \code{threshold = 0.25}, and is ignored if \code{EV} is provided.}

\item{family}{The likelihood function for the outcome variable. Current options are \code{family = gaussian()}, \code{student_t()} and \code{poisson(link = "log")}, and \code{binomial(link = "logit")}.}

\item{prior}{A named list of parameters for prior distributions (see \code{\link[geostan]{priors}}):  \describe{

\item{intercept}{The intercept is assigned a Gaussian prior distribution (see \code{\link[geostan]{normal}}}.

\item{beta}{Regression coefficients are assigned Gaussian prior distributions. Variables must follow their order of appearance in the model \code{formula}. Note that if you also use \code{slx} terms (spatially lagged covariates), and you use custom priors for \code{beta}, then you have to provide priors for the slx terms. Since slx terms are \emph{prepended} to the design matrix, the prior for the slx term will be listed first.
}

\item{sigma}{For \code{family = gaussian()} and \code{family = student_t()} models, the scale parameter, \code{sigma}, is assigned a (half-) Student's t prior distribution. The half-Student's t prior for \code{sigma} is constrained to be positive.}

\item{nu}{\code{nu} is the degrees of freedom parameter in the Student's t likelihood (only used when \code{family = student_t()}). \code{nu} is assigned a gamma prior distribution. The default prior is \code{prior = list(nu = gamma(alpha = 3, beta = 0.2))}. }

\item{tau}{The scale parameter for random effects, or varying intercepts, terms. This scale parameter, \code{tau}, is assigned a half-Student's t prior. To set this, use, e.g., \code{prior = list(tau = student_t(df = 20, location = 0, scale = 20))}.}

\item{beta_ev}{The eigenvector coefficients are assigned the horseshoe prior (Piironen and Vehtari, 2017), parameterized by \code{global_scale} (to control overall prior sparsity), plus the degrees of freedom and scale of a Student's t model for any large coefficients (see \code{\link[geostan]{priors}}). To allow the spatial filter to account for a greater amount of spatial autocorrelation (i.e., if you find the residuals contain spatial autocorrelation), increase the global scale parameter (to a maximum of \code{global_scale = 1}).}
}}

\item{ME}{To model observational uncertainty (i.e. measurement or sampling error) in any or all of the covariates, provide a list of data as constructed by the \code{\link[geostan]{prep_me_data}} function.}

\item{centerx}{To center predictors on their mean values, use \code{centerx = TRUE}. If the ME argument is used, the modeled covariate (i.e., latent variable), rather than the raw observations, will be centered. When using the ME argument, this is the recommended method for centering the covariates.}

\item{censor_point}{Integer value indicating the maximum censored value; this argument is for modeling censored (suppressed) outcome data, typically disease case counts or deaths. For example, the US Centers for Disease Control and Prevention censors (does not report) death counts that are nine or fewer, so if you're using CDC WONDER mortality data you could provide \code{censor_point = 9}.}

\item{prior_only}{Draw samples from the prior distributions of parameters only.}

\item{chains}{Number of MCMC chains to estimate. Default \code{chains = 4}.}

\item{iter}{Number of samples per chain. Default \code{iter = 2000}.}

\item{refresh}{Stan will print the progress of the sampler every \code{refresh} number of samples. Defaults to \code{500}; set \code{refresh=0} to silence this.}

\item{keep_all}{If \code{keep_all = TRUE} then samples for all parameters in the Stan model will be kept; this is necessary if you want to do model comparison with Bayes factors and the \code{bridgesampling} package.}

\item{pars}{Optional; specify any additional parameters you'd like stored from the Stan model.}

\item{control}{A named list of parameters to control the sampler's behavior. See \link[rstan]{stan} for details.}

\item{...}{Other arguments passed to \link[rstan]{sampling}.}
}
\value{
An object of class class \code{geostan_fit} (a list) containing:
\describe{
\item{summary}{Summaries of the main parameters of interest; a data frame}
\item{diagnostic}{Widely Applicable Information Criteria (WAIC) with a measure of effective number of parameters (\code{eff_pars}) and mean log pointwise predictive density (\code{lpd}), and mean residual spatial autocorrelation as measured by the Moran coefficient.}
\item{data}{a data frame containing the model data}
\item{EV}{A matrix of eigenvectors created with \code{w} and \code{geostan::make_EV}}
\item{C}{The spatial weights matrix used to construct EV}
\item{family}{the user-provided or default \code{family} argument used to fit the model}
\item{formula}{The model formula provided by the user (not including ESF component)}
\item{slx}{The \code{slx} formula}
\item{re}{A list containing \code{re},  the random effects (varying intercepts) formula if provided, and
\code{data} a data frame with columns \code{id}, the grouping variable, and \code{idx}, the index values assigned to each group.}
\item{priors}{Prior specifications.}
\item{x_center}{If covariates are centered internally (\code{centerx = TRUE}), then \code{x_center} is a numeric vector of the values on which covariates were centered.}
\item{ME}{The \code{ME} data list, if one was provided by the user for measurement error models.}
\item{spatial}{A data frame with the name of the spatial component parameter ("esf") and method ("ESF")}
\item{stanfit}{an object of class \code{stanfit} returned by \code{rstan::stan}}
}
}
\description{
Fit a spatial regression model using eigenvector spatial filtering (ESF).
}
\details{
Eigenvector spatial filtering (ESF) is a method for spatial regression analysis. ESF is extensively covered in Griffith et al. (2019). This function implements the methodology introduced in Donegan et al. (2020), which uses Piironen and Vehtari's (2017) regularized horseshoe prior.

ESF decomposes spatial autocorrelation into a linear combination of various patterns, typically at different scales (such as local, regional, and global trends). By adding a spatial filter to a regression model, any spatial autocorrelation is shifted from the residuals to the spatial filter. ESF models take the spectral decomposition of a transformed spatial connectivity matrix, \code{C}. The resulting eigenvectors, \code{EV}, are mutually orthogonal and uncorrelated map patterns. The spatial filter is \code{EV * beta_ev}, where \code{beta_ev} is a vector of coefficients.

ESF decomposes the data into a global mean, \code{alpha}, global patterns contributed by covariates, \code{X * beta}, spatial trends, \code{EV * beta_ev}, and residual variation. Thus, for \code{family=gaussian()},\preformatted{Y ~ Gauss(alpha + X * beta + EV * beta_ev, sigma).
}

An ESF component can be incorporated into the linear predictor of any generalized linear model. For example, a spatial Poisson model for rare disease incidence may be specified as follows:\preformatted{Y ~ Poisson(exp(offset + Mu))
Mu = alpha + EV * beta_ev + A
A ~ Guass(0, tau)
tau ~ student(20, 0, 2)
beta_ev ~ horseshoe(.)
}

The \code{\link[geostan]{spatial.geostan_fit}} method will return \code{EV * beta}.

The model can also be extended to the space-time domain; see \link[geostan]{shape2mat} to specify a space-time connectivity matrix.

The coefficients \code{beta_ev} are assigned the regularized horseshoe prior (Piironen and Vehtari, 2017), resulting in a relatively sparse model specification. In addition, numerous eigenvectors are automatically dropped because they represent trace amounts of spatial autocorrelation (this is controlled by the \code{threshold} argument). By default, \code{stan_esf} will drop all eigenvectors representing negative spatial autocorrelation patterns. You can change this behavior using the \code{nsa} argument.
\subsection{Spatially lagged covariates (SLX)}{

The \code{slx} argument is a convenience function for including SLX terms. For example,\preformatted{stan_glm(y ~ x1 + x2, slx = ~ x1, ...)
}

is a shortcut for\preformatted{stan_glm(y ~ I(W \%*\% x1) + x1 + x2, ...)
}

where \code{W} is a row-standardized spatial weights matrix (see \code{\link[geostan]{shape2mat}}). SLX terms will always be \emph{prepended} to the design matrix, as above, which is important to know when setting prior distributions for regression coefficients.

For measurement error (ME) models, the SLX argument is the only way to include spatially lagged covariates since the SLX term needs to be re-calculated on each iteration of the MCMC algorithm.
}

\subsection{Measurement error (ME) models}{

The ME models are designed for surveys with spatial sampling designs, such as the American Community Survey (ACS) estimates (Donegan et al. 2021; Donegan 2021). With estimates, \code{x}, and their standard errors, \code{se}, the ME models have one of the the following two specifications, depending on the user input:\preformatted{x ~ Gauss(x_true, se)
x_true ~ MVGauss(mu, Sigma)
Sigma = (I - rho * C)^(-1) M * tau^2
mu ~ Gauss(0, 100)
tau ~ student_t(10, 0, 40)
rho ~ uniform(lower_bound, upper_bound)
}

where the covariance matrix, \code{Sigma}, has the conditional autoregressive specification, and \code{tau} is the scale parameter. For non-spatial ME models, the following is used instead:\preformatted{x ~ Gauss(x_true, se)
x_true ~ student_t(df, mu, sigma)
df ~ gamma(3, 0.2)
mu ~ Gauss(0, 100)
sigma ~ student_t(10, 0, 40)
}

For strongly skewed variables, such census tract poverty rates, it can be advantageous to apply a logit transformation to \code{x_true} before applying the CAR or Student t prior model. When the \code{logit} argument is used, the model becomes:\preformatted{x ~ Gauss(x_true, se)
logit(x_true) ~ MVGauss(mu, Sigma)
}

and similar for the Student t model.
}

\subsection{Censored counts}{

Vital statistics systems and disease surveillance programs typically suppress case counts when they are smaller than a specific threshold value. In such cases, the observation of a censored count is not the same as a missing value; instead, you are informed that the value is an integer somewhere between zero and the threshold value. For Poisson models (\verb{family = poisson())}), you can use the \code{censor_point} argument to encode this information into your model.

Internally, \code{geostan} will keep the index values of each censored observation, and the index value of each of the fully observed outcome values. For all observed counts, the likelihood statement will be:\preformatted{p(y_i | data, model) = Poisson(y_i | fitted_i), 
}

as usual. For each censored count, the likelihood statement will equal the cumulative Poisson distribution function for values zero through the censor point:\preformatted{p(y_j | data, model) = sum_\{m=0\}^censor_point Poisson( c_m | fitted_j),
}

For example, the US Centers for Disease Control and Prevention's CDC WONDER database censors all death counts between 0 and 9. To model CDC WONDER mortality data, you could provide \code{censor_point = 9} and then the likelihood statement for censored counts would equal the summation of the Poisson probability mass function over each integer ranging from zero through 9 (inclusive), conditional on the fitted values (i.e., all model parameters). See Donegan (2021) for additional discussion, references, and Stan code.
}
}
\examples{
\donttest{
data(sentencing)
# spatial weights matrix with binary coding scheme
C <- shape2mat(sentencing, style = "B")

# log-expected number of sentences
## expected counts are based on county racial composition and mean sentencing rates
log_e <- log(sentencing$expected_sents)

# fit spatial Poisson model with ESF + unstructured 'random effects'
fit.esf <- stan_esf(sents ~ offset(log_e),
                   re = ~ name,
                   family = poisson(),
                   data = sentencing,
                   C = C,
                   chains = 2, iter = 800) # for speed only

# spatial diagnostics 
sp_diag(fit.esf, sentencing)
plot(fit.esf)

# plot marginal posterior distributions of beta_ev (eigenvector coefficients)
plot(fit.esf, pars = "beta_ev")

# plot the marginal posterior distributions of the spatial filter 
plot(fit.esf, pars = "esf")

# calculate log-standardized incidence ratios 
library(ggplot2)
library(sf)
f <- fitted(fit.esf, rates = FALSE)$mean
SSR <-  f / sentencing$expected_sents
log.SSR <- log( SSR, base = 2 )

# map the log-SSRs
st_as_sf(sentencing) \%>\%
 ggplot() +
 geom_sf(aes(fill = log.SSR)) +
 scale_fill_gradient2(
   midpoint = 0,
   name = NULL,
   breaks = seq(-3, 3, by = 0.5)
 ) +
 labs(title = "Log-Standardized Sentencing Ratios",
      subtitle = "log( Fitted/Expected ), base 2"
 ) +
 theme_void()
}
}
\author{
Connor Donegan, \email{Connor.Donegan@UTDallas.edu}
}
