\name{pathInc}
\alias{pathInc}
\alias{pathInc,TransitionLayer,Coords,Coords,missing,missing,missing-method}
\alias{pathInc,TransitionLayer,Coords,Coords,missing,numeric,missing-method}
\alias{pathInc,TransitionLayer,Coords,Coords,missing,missing,Transition-method}
\alias{pathInc,TransitionLayer,Coords,Coords,missing,numeric,Transition-method}
\alias{pathInc,TransitionLayer,Coords,Coords,Coords,missing,missing-method}
\alias{pathInc,TransitionLayer,Coords,Coords,Coords,numeric,missing-method}
\alias{pathInc,TransitionLayer,Coords,Coords,Coords,missing,Transition-method}
\alias{pathInc,TransitionLayer,Coords,Coords,Coords,numeric,Transition-method}

\title{Incidence of paths from a common origin: overlap and non-overlap}
\description{
Calculate the overlap and non-overlap of paths departing from a common origin. Two algorithms are available: random walk and randomised shortest paths.}
\usage{
pathInc(x, origin, from, to, theta, weight, ...)

}
\arguments{
  \item{x}{transition matrix (class Transition)}
  \item{origin}{coordinates of the origin (one point location, SpatialPoints, matrix or numeric class)}  
  \item{from}{coordinates of the destinations (SpatialPoints, matrix or numeric class)}
  \item{to}{second set of coordinates of the destinations (can be missing)}
  \item{theta}{value > 0 and < 20 (randomised shortest paths) or missing (random walk)}
  \item{weight}{weight matrix -- Reciprocals of the non-zero values are used as weights. If missing, reciprocals of the transition matrix are used.}
  \item{...}{an extra argument for functions to be defined manually (see below)}
}
\details{
This is a complex wrapper function that calculates to what extent dispersal routes overlap or do not overlap.

First, the function calculates the trajectories for all "from" and "to" locations, starting from a single "origin" location. These trajectories can either be based on random walks or randomised shortest paths (giving a value to theta).

Then, for all unique pairs of trajectories, it calculates the extent to which these trajectories overlap or diverge. These values are given back to the user as a list of (distance) matrices.

If only "from" coordinates are given, the function calculates symmetric distance matrices for all combinations of points in "from". If both "from" and "to" coordinates are given, the function calculates a matrix of values with rows for all locations in "from" and columns for all locations in "to".

Overlap is currently calculated as the minimum values of each pair of trajectories compared. Non-overlap uses the following formula: Nonoverlap = max(0,max(a,b)*(1-min(a,b))-min(a,b)) (see van Etten and Hijmans 2010). 

}
\value{
list of dist objects or list of matrices
}
\references{
McRae B.H., B.G. Dickson, and T. Keitt. 2008. Using circuit theory to model connectivity in ecology, evolution, and conservation. \emph{Ecology} 89:2712-2724.
  
Saerens M., L. Yen, F. Fouss, and Y. Achbany. 2009. Randomized shortest-path problems: two related models. \emph{Neural Computation}, 21(8):2363-2404.

van Etten, J., and R.J. Hijmans. 2010. A geospatial modelling approach integrating archaeobotany and genetics to trace the origin and dispersal of domesticated plants. \emph{PLoS ONE} 5(8): e12060.
}
\author{ Jacob van Etten \email{jacobvanetten@yahoo.com}. Implementation of randomised shortest paths based on Matlab code by Marco Saerens. }
\examples{
#Create TransitionLayer
r <- raster(ncol=36,nrow=18)
r <- setValues(r,rep(1,times=ncell(r)))
tr <- transition(r,mean,directions=4)

#Two different types of correction are required
trR <- geoCorrection(tr, type="r", multpl=FALSE)
trC <- geoCorrection(tr, type="c", multpl=FALSE)

#Create TransitionStack
ts <- stack(trR, trR)

#Points for origin and coordinates between which to calculate path (non)overlaps
sP0 <- SpatialPoints(cbind(0,0))
sP1 <- SpatialPoints(cbind(c(65,5,-65),c(-55,35,-35)))

#Randomised shortest paths
#rescaling is needed: exp(-theta * trC) should give reasonable values
trC <- trC / median(transitionMatrix(trC)@x) #divide by median of the non-zero values
pathInc(trC, origin=sP0, from=sP1, theta=2)

#Random walk
pathInc(trR, origin=sP0, from=sP1)

#TransitionStack as weights
pathInc(trR, origin=sP0, from=sP1, weight=ts)

#Demonstrate use of an alternative function
#The current default is to take the minimum of each pair of layers

altoverlap <- function(a, b)
{
	aV <- as.vector(a[,rep(1:ncol(a), each=ncol(b))])
	bV <- as.vector(b[,rep(1:ncol(b), times=ncol(a))])
	result <- matrix(aV * bV, nrow = nrow(a), ncol=ncol(a)*ncol(b))
	return(result)
}

pathInc(trR, origin=sP0, from=sP1, weight=ts, functions=list(altoverlap))

}
\keyword{ spatial }

