% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdalraster_proc.R
\name{rasterFromRaster}
\alias{rasterFromRaster}
\title{Create a raster from an existing raster as template}
\usage{
rasterFromRaster(
  srcfile,
  dstfile,
  fmt = NULL,
  nbands = NULL,
  dtName = NULL,
  options = NULL,
  init = NULL,
  dstnodata = init
)
}
\arguments{
\item{srcfile}{Source raster filename.}

\item{dstfile}{Output raster filename.}

\item{fmt}{Output raster format name (e.g., "GTiff" or "HFA"). Will attempt
to guess from the output filename if \code{fmt} is not specified.}

\item{nbands}{Number of output bands.}

\item{dtName}{Output raster data type name. Commonly used types include
\code{"Byte"}, \code{"Int16"}, \code{"UInt16"}, \code{"Int32"} and \code{"Float32"}.}

\item{options}{Optional list of format-specific creation options in a
vector of "NAME=VALUE" pairs
(e.g., \code{options = c("COMPRESS=LZW")} to set LZW compression
during creation of a GTiff file).}

\item{init}{Numeric value to initialize all pixels in the output raster.}

\item{dstnodata}{Numeric nodata value for the output raster.}
}
\value{
Returns the destination filename invisibly.
}
\description{
\code{rasterFromRaster()} creates a new raster with spatial reference,
extent and resolution taken from a template raster, without copying data.
Optionally changes the format, number of bands, data type and nodata value,
sets driver-specific dataset creation options, and initializes to a value.
}
\examples{
## band 2 in a FARSITE landscape file has slope degrees
## convert slope degrees to slope percent in a new raster
lcp_file <- system.file("extdata/storm_lake.lcp", package="gdalraster")
ds_lcp <- new(GDALRaster, lcp_file, read_only=TRUE)
ds_lcp$getMetadata(band=2, domain="")

slpp_file <- paste0(tempdir(), "/", "storml_slpp.tif")
opt = c("COMPRESS=LZW")
rasterFromRaster(srcfile = lcp_file,
                 dstfile = slpp_file,
                 nbands = 1,
                 dtName = "Int16",
                 options = opt,
                 init = -32767)
ds_slp <- new(GDALRaster, slpp_file, read_only=FALSE)

## slpp_file is initialized to -32767 and nodata value set
ds_slp$getNoDataValue(band=1)

## extent and cell size are the same as lcp_file
ds_lcp$bbox()
ds_lcp$res()
ds_slp$bbox()
ds_slp$res()

## convert slope degrees in lcp_file band 2 to slope percent in slpp_file
## bring through LCP nodata -9999 to the output nodata value
ncols <- ds_slp$getRasterXSize()
nrows <- ds_slp$getRasterYSize()
for (row in 0:(nrows-1)) {
    rowdata <- ds_lcp$read(band=2, 
                           xoff=0, yoff=row,
                           xsize=ncols, ysize=1,
                           out_xsize=ncols, out_ysize=1)
    rowslpp <- tan(rowdata*pi/180) * 100
    rowslpp[rowdata==-9999] <- -32767
    dim(rowslpp) <- c(1, ncols)
    ds_slp$write(band=1, xoff=0, yoff=row, xsize=ncols, ysize=1, rowslpp)
}

## min, max, mean, sd
ds_slp$getStatistics(band=1, approx_ok=FALSE, force=TRUE)

ds_slp$close()
ds_lcp$close()
}
\seealso{
\code{\link[=GDALRaster]{GDALRaster-class}}, \code{\link[=create]{create()}}, \code{\link[=createCopy]{createCopy()}},
\code{\link[=bandCopyWholeRaster]{bandCopyWholeRaster()}}
}
