#' Species lists
#'
#' While there are reasons why users may need to check every record meeting their
#' search criteria (i.e. using \code{\link{ala_occurrences}}), a common use case
#' is to simply identify which species occur in a specified region, time period,
#' or taxonomic group. This function returns a \code{data.frame} with one row
#' per species, and columns giving associated taxonomic information.
#'
#' @param taxa \code{data.frame}: generated by a call to \code{\link{select_taxa}()}. This
#' argument also accepts a vector of unique species identifiers.
#' @param filters \code{data.frame}: generated by a call to \code{\link{select_filters}()}
#' @param locations \code{string}: generated by a call to \code{\link{select_locations}()}
#' @return A \code{data.frame} of matching species.
#' @details
#' The primary use case of this function is to extract species-level information
#' given a set of criteria defined by \code{\link{select_taxa}()},
#' \code{\link{select_filters}()} or \code{\link{select_locations}()}. If the purpose
#' is simply to get taxonomic information that is not restricted by filtering,
#' then \code{\link{select_taxa}()} is more efficient. Similarly, if counts are
#' required that include filters but without returning taxonomic detail, then
#' \code{\link{ala_counts}()} is more efficient (see examples).
#' @examples \dontrun{
#'
#' # Lookup genus "Heleioporus" in the ALA
#' select_taxa("Heleioporus")
#'
#' # How many records are there for this genus?
#' ala_counts(select_taxa("Heleioporus"))
#' # or equivalently:
#' select_taxa("Heleioporus", counts = TRUE)
#'
#' # How best to get taxonomic info on species within this genus?
#' # also includes a row for genus (i.e. not just species)
#' select_taxa("Heleioporus", children = TRUE)
#' # returns counts by species, but no taxonomic information
#' ala_counts(select_taxa("Heleioporus"), group_by = "species")
#' # every row is a species with associated taxonomic data
#' ala_species(select_taxa("Heleioporus"))
#' }
#' @export ala_species

# If the facet search download worked properly, this should also return counts. But, as this
# function is likely to be used to download long species lists, for now we will make do
# without the counts- otherwise will require lots of pagination.
ala_species <- function(taxa, filters, locations) {
  # Use facet search with additional options

  url <- getOption("galah_server_config")$base_url_biocache

  query <- list()

  if (missing(taxa) & missing(filters) & missing(locations)) {
    warning("This query will return a list of all species in the ALA")
  }

  if (missing(taxa)) { taxa <- NULL }
  if (missing(filters)) { filters <- NULL }
  if (missing(locations)) { locations <- NULL }
  
  query <- build_query(taxa, filters, locations)

  query$facets <- "species_guid"
  query$lookup  <- "true"
  
  path <- "ws/occurrences/facets/download"

  cache_file <- cache_filename(c(url, path, unlist(query)), ext = ".csv")

  caching <- getOption("galah_config")$caching

  if (caching && file.exists(cache_file)) {
    message("Using cached file")
    return(read.csv(cache_file))
  }

  if (!caching & !dir.exists(ala_config()$cache_directory)) {
    dir.create(ala_config()$cache_directory)
    file.create(cache_file)
  }
  data <- ala_download(url, path = path, params = query,
                       cache_file = cache_file)
  # overwrite file with fixed names
  names(data) <- rename_columns(names(data), type = "checklist")

  if (caching) {
    write.csv(data, cache_file, row.names = FALSE)
  } else {
    file.remove(cache_file)
  }
  return(data)
}